"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const bech32_1 = require("./bech32");
const hex_1 = require("./hex");
describe("bech32", () => {
    // test data generate using https://github.com/nym-zone/bech32
    // bech32 -e -h eth 9d4e856e572e442f0a4b2763e72d08a0e99d8ded
    const ethAddressRaw = (0, hex_1.fromHex)("9d4e856e572e442f0a4b2763e72d08a0e99d8ded");
    describe("toBech32", () => {
        it("works", () => {
            expect((0, bech32_1.toBech32)("eth", ethAddressRaw)).toEqual("eth1n48g2mjh9ezz7zjtya37wtgg5r5emr0drkwlgw");
        });
        it("works for very short data", () => {
            expect(() => (0, bech32_1.toBech32)("eth", new Uint8Array(1))).not.toThrow();
        });
        it("works for very long prefixes", () => {
            expect(() => (0, bech32_1.toBech32)("p".repeat(70), new Uint8Array(20))).toThrowError(/exceeds length limit/i);
        });
        // See https://github.com/bitcoin/bips/blob/master/bip-0173.mediawiki#Bech32
        it("works if result is 90 characters", () => {
            const result = (0, bech32_1.toBech32)("eth", new Uint8Array(50));
            expect(result.length).toEqual(90);
        });
        it("throws if result exceeds 90 characters", () => {
            expect(() => (0, bech32_1.toBech32)("eth", new Uint8Array(51))).toThrowError(/exceeds length limit/i);
        });
        it("works if a limit parameter is provided", () => {
            const limit = 1024;
            const result = (0, bech32_1.toBech32)("eth", new Uint8Array(51), limit);
            expect(result).toEqual("eth1qqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqug55er");
            expect(result.length).toBeGreaterThan(90);
        });
        it("throws if result exceeds the provided limit parameter", () => {
            const limit = 10;
            expect(() => (0, bech32_1.toBech32)("eth", ethAddressRaw, limit)).toThrowError(/exceeds length limit/i);
        });
    });
    describe("fromBech32", () => {
        it("works", () => {
            expect((0, bech32_1.fromBech32)("eth1n48g2mjh9ezz7zjtya37wtgg5r5emr0drkwlgw")).toEqual({
                prefix: "eth",
                data: ethAddressRaw,
            });
        });
        it("works for upper case address", () => {
            // "For presentation, lowercase is usually preferable, but inside QR codes uppercase SHOULD be used, as those permit the use of alphanumeric mode, which is 45% more compact than the normal byte mode."
            // https://github.com/bitcoin/bips/blob/master/bip-0173.mediawiki
            expect((0, bech32_1.fromBech32)("ETH1N48G2MJH9EZZ7ZJTYA37WTGG5R5EMR0DRKWLGW")).toEqual({
                prefix: "eth",
                data: ethAddressRaw,
            });
        });
        it("works for addresses which exceed the specification limit of 90 characters", () => {
            // Example from https://github.com/cosmos/cosmos-sdk/pull/6237#issuecomment-658116534
            expect(() => (0, bech32_1.fromBech32)("cosmospub1ytql0csgqvfzd666axrjzqmn5q2ucztcyxw8hvlzen94ay05tegaerkug5pn3xn8wqdymt598ufzd666axrjzqsxllmwacap3f6xyc4x30jl8ecrcs2tze3zzgxkmthcsqxnqxhwwgfzd666axrjzqs2rlu3wz5gnslgpprszjr8r65n0d6y39q657th77eyvengtk3z0y6h2pnk")).not.toThrow();
        });
        it("throws for addresses which exceed the specification limit of 90 characters if a limit is specified", () => {
            // Example from https://github.com/cosmos/cosmos-sdk/pull/6237#issuecomment-658116534
            expect(() => (0, bech32_1.fromBech32)("cosmospub1ytql0csgqvfzd666axrjzqmn5q2ucztcyxw8hvlzen94ay05tegaerkug5pn3xn8wqdymt598ufzd666axrjzqsxllmwacap3f6xyc4x30jl8ecrcs2tze3zzgxkmthcsqxnqxhwwgfzd666axrjzqs2rlu3wz5gnslgpprszjr8r65n0d6y39q657th77eyvengtk3z0y6h2pnk", 90)).toThrowError(/exceeds length limit/i);
        });
        it("throws for mixed case addresses", () => {
            // "Decoders MUST NOT accept strings where some characters are uppercase and some are lowercase (such strings are referred to as mixed case strings)."
            // https://github.com/bitcoin/bips/blob/master/bip-0173.mediawiki
            expect(() => (0, bech32_1.fromBech32)("Eth1n48g2mjh9ezz7zjtya37wtgg5r5emr0drkwlgw")).toThrowError(/Mixed-case/i);
            expect(() => (0, bech32_1.fromBech32)("eTh1n48g2mjh9ezz7zjtya37wtgg5r5emr0drkwlgw")).toThrowError(/Mixed-case/i);
            expect(() => (0, bech32_1.fromBech32)("ETH1n48g2mjh9ezz7zjtya37wtgg5r5emr0drkwlgw")).toThrowError(/Mixed-case/i);
            expect(() => (0, bech32_1.fromBech32)("eth1n48g2mjh9Ezz7zjtya37wtgg5r5emr0drkwlgw")).toThrowError(/Mixed-case/i);
        });
    });
    describe("normalizeBech32", () => {
        it("works", () => {
            expect((0, bech32_1.normalizeBech32)("eth1n48g2mjh9ezz7zjtya37wtgg5r5emr0drkwlgw")).toEqual("eth1n48g2mjh9ezz7zjtya37wtgg5r5emr0drkwlgw");
            expect((0, bech32_1.normalizeBech32)("ETH1N48G2MJH9EZZ7ZJTYA37WTGG5R5EMR0DRKWLGW")).toEqual("eth1n48g2mjh9ezz7zjtya37wtgg5r5emr0drkwlgw");
        });
        it("throws for mixed case addresses", () => {
            // "Decoders MUST NOT accept strings where some characters are uppercase and some are lowercase (such strings are referred to as mixed case strings)."
            // https://github.com/bitcoin/bips/blob/master/bip-0173.mediawiki
            expect(() => (0, bech32_1.normalizeBech32)("Eth1n48g2mjh9ezz7zjtya37wtgg5r5emr0drkwlgw")).toThrowError(/Mixed-case/i);
            expect(() => (0, bech32_1.normalizeBech32)("eTh1n48g2mjh9ezz7zjtya37wtgg5r5emr0drkwlgw")).toThrowError(/Mixed-case/i);
            expect(() => (0, bech32_1.normalizeBech32)("ETH1n48g2mjh9ezz7zjtya37wtgg5r5emr0drkwlgw")).toThrowError(/Mixed-case/i);
            expect(() => (0, bech32_1.normalizeBech32)("eth1n48g2mjh9Ezz7zjtya37wtgg5r5emr0drkwlgw")).toThrowError(/Mixed-case/i);
        });
    });
});
//# sourceMappingURL=bech32.spec.js.map