"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const coin_pretty_1 = require("./coin-pretty");
const decimal_1 = require("./decimal");
const int_1 = require("./int");
const coin_utils_1 = require("./coin-utils");
const dec_utils_1 = require("./dec-utils");
describe("Test CoinPretty", () => {
    it("Test creation of CoinPretty", () => {
        expect(new coin_pretty_1.CoinPretty({
            coinDenom: "ATOM",
            coinMinimalDenom: "uatom",
            coinDecimals: 6,
        }, new int_1.Int(1234))
            .toDec()
            .equals(new decimal_1.Dec("0.001234"))).toBe(true);
        expect(new coin_pretty_1.CoinPretty({
            coinDenom: "ATOM",
            coinMinimalDenom: "uatom",
            coinDecimals: 6,
        }, new int_1.Int(1234)).toString()).toBe("0.001234 ATOM");
        expect(new coin_pretty_1.CoinPretty({
            coinDenom: "ATOM",
            coinMinimalDenom: "uatom",
            coinDecimals: 6,
        }, 1234)
            .toDec()
            .equals(new decimal_1.Dec("0.001234"))).toBe(true);
        expect(new coin_pretty_1.CoinPretty({
            coinDenom: "ATOM",
            coinMinimalDenom: "uatom",
            coinDecimals: 6,
        }, 1234).toString()).toBe("0.001234 ATOM");
        expect(new coin_pretty_1.CoinPretty({
            coinDenom: "ATOM",
            coinMinimalDenom: "uatom",
            coinDecimals: 6,
        }, "1234.5")
            .toDec()
            .equals(new decimal_1.Dec("0.0012345"))).toBe(true);
        expect(new coin_pretty_1.CoinPretty({
            coinDenom: "ATOM",
            coinMinimalDenom: "uatom",
            coinDecimals: 6,
        }, "1234.5").toString()).toBe("0.001234 ATOM");
        expect(new coin_pretty_1.CoinPretty({
            coinDenom: "ATOM",
            coinMinimalDenom: "uatom",
            coinDecimals: 6,
        }, new decimal_1.Dec("1234.5"))
            .toDec()
            .equals(new decimal_1.Dec("0.0012345"))).toBe(true);
        expect(new coin_pretty_1.CoinPretty({
            coinDenom: "ATOM",
            coinMinimalDenom: "uatom",
            coinDecimals: 6,
        }, new decimal_1.Dec("1234.5")).toString()).toBe("0.001234 ATOM");
    });
    it("Basic test for CoinPretty", () => {
        const pretty = new coin_pretty_1.CoinPretty({
            coinDenom: "ATOM",
            coinMinimalDenom: "uatom",
            coinDecimals: 6,
        }, new int_1.Int(1234));
        expect(pretty.toDec().equals(new decimal_1.Dec("0.001234"))).toBeTruthy();
        expect(pretty.toString()).toBe("0.001234 ATOM");
        expect(pretty.separator("").toString()).toBe("0.001234ATOM");
        expect(pretty.increasePrecision(1).toString()).toBe("0.000123 ATOM");
        expect(pretty.decreasePrecision(1).toString()).toBe("0.012340 ATOM");
        expect(pretty.decreasePrecision(1).trim(true).toString()).toBe("0.01234 ATOM");
        expect(pretty.moveDecimalPointLeft(1).toString()).toBe("0.000123 ATOM");
        expect(pretty.moveDecimalPointRight(1).toString()).toBe("0.012340 ATOM");
        expect(pretty.moveDecimalPointRight(1).trim(true).toString()).toBe("0.01234 ATOM");
        expect(pretty.maxDecimals(7).add(new decimal_1.Dec("0.1")).toString()).toBe("0.0012341 ATOM");
        expect(pretty.maxDecimals(7).sub(new decimal_1.Dec("0.1")).toString()).toBe("0.0012339 ATOM");
        expect(pretty.maxDecimals(7).mul(new decimal_1.Dec("0.1")).toString()).toBe("0.0001234 ATOM");
        expect(pretty.maxDecimals(7).quo(new decimal_1.Dec("0.1")).toString()).toBe("0.0123400 ATOM");
        expect(pretty
            .add(new coin_pretty_1.CoinPretty({
            coinDenom: "ATOM",
            coinMinimalDenom: "uatom",
            coinDecimals: 6,
        }, new int_1.Int(1200000)))
            .toString()).toBe("1.201234 ATOM");
        expect(pretty
            .sub(new coin_pretty_1.CoinPretty({
            coinDenom: "ATOM",
            coinMinimalDenom: "uatom",
            coinDecimals: 6,
        }, new int_1.Int(1200000)))
            .toString()).toBe("-1.198766 ATOM");
        // If target is `CoinPretty` and it has different denom, do nothing.
        expect(pretty
            .add(new coin_pretty_1.CoinPretty({
            coinDenom: "SCRT",
            coinMinimalDenom: "uscrt",
            coinDecimals: 6,
        }, new int_1.Int(1200000)))
            .toString()).toBe("0.001234 ATOM");
        // If target is `CoinPretty` and it has different denom, do nothing.
        expect(pretty
            .sub(new coin_pretty_1.CoinPretty({
            coinDenom: "SCRT",
            coinMinimalDenom: "uscrt",
            coinDecimals: 6,
        }, new int_1.Int(1200000)))
            .toString()).toBe("0.001234 ATOM");
    });
    it("Basic test for CoinPretty 2", () => {
        const pretty = new coin_pretty_1.CoinPretty({
            coinDenom: "ATOM",
            coinMinimalDenom: "uatom",
            coinDecimals: 6,
        }, new decimal_1.Dec("12.1234"));
        expect(pretty.toString()).toBe("0.000012 ATOM");
        expect(pretty.maxDecimals(10).toString()).toBe("0.0000121234 ATOM");
        expect(pretty.increasePrecision(1).toString()).toBe("0.000001 ATOM");
        expect(pretty.decreasePrecision(1).toString()).toBe("0.000121 ATOM");
        expect(pretty.moveDecimalPointLeft(1).toString()).toBe("0.000001 ATOM");
        expect(pretty.moveDecimalPointRight(1).toString()).toBe("0.000121 ATOM");
        expect(pretty.maxDecimals(7).add(new decimal_1.Dec("0.1")).toString()).toBe("0.0000122 ATOM");
        expect(pretty.maxDecimals(7).sub(new decimal_1.Dec("0.1")).toString()).toBe("0.0000120 ATOM");
        expect(pretty.maxDecimals(7).mul(new decimal_1.Dec("0.1")).toString()).toBe("0.0000012 ATOM");
        expect(pretty.maxDecimals(7).quo(new decimal_1.Dec("0.1")).toString()).toBe("0.0001212 ATOM");
        expect(pretty
            .add(new coin_pretty_1.CoinPretty({
            coinDenom: "ATOM",
            coinMinimalDenom: "uatom",
            coinDecimals: 6,
        }, new int_1.Int(1200000)))
            .toString()).toBe("1.200012 ATOM");
        expect(pretty
            .sub(new coin_pretty_1.CoinPretty({
            coinDenom: "ATOM",
            coinMinimalDenom: "uatom",
            coinDecimals: 6,
        }, new int_1.Int(1200000)))
            .toString()).toBe("-1.199987 ATOM");
    });
    it("Test toCoin() for CoinPretty", () => {
        expect(new coin_pretty_1.CoinPretty({
            coinDenom: "ATOM",
            coinMinimalDenom: "uatom",
            coinDecimals: 6,
        }, new decimal_1.Dec("0.1234")).toCoin()).toStrictEqual({
            denom: "uatom",
            amount: "0",
        });
        expect(new coin_pretty_1.CoinPretty({
            coinDenom: "ATOM",
            coinMinimalDenom: "uatom",
            coinDecimals: 6,
        }, new decimal_1.Dec("12.1234")).toCoin()).toStrictEqual({
            denom: "uatom",
            amount: "12",
        });
        expect(new coin_pretty_1.CoinPretty({
            coinDenom: "ATOM",
            coinMinimalDenom: "uatom",
            coinDecimals: 6,
        }, new decimal_1.Dec("123456.1234")).toCoin()).toStrictEqual({
            denom: "uatom",
            amount: "123456",
        });
        expect(new coin_pretty_1.CoinPretty({
            coinDenom: "ATOM",
            coinMinimalDenom: "uatom",
            coinDecimals: 6,
        }, new int_1.Int("12345600")).toCoin()).toStrictEqual({
            denom: "uatom",
            amount: "12345600",
        });
    });
    it("Test CoinPretty's setCurrency", () => {
        const pretty = new coin_pretty_1.CoinPretty({
            coinDenom: "ATOM",
            coinMinimalDenom: "uatom",
            coinDecimals: 6,
        }, new int_1.Int("123456"));
        expect(pretty
            .setCurrency({
            coinDenom: "TEST",
            coinMinimalDenom: "utest",
            coinDecimals: 6,
        })
            .toString()).toBe("0.123456 TEST");
        expect(pretty
            .setCurrency({
            coinDenom: "TEST",
            coinMinimalDenom: "utest",
            coinDecimals: 3,
        })
            .toString()).toBe("123.456000 TEST");
        expect(pretty
            .setCurrency({
            coinDenom: "TEST",
            coinMinimalDenom: "utest",
            coinDecimals: 3,
        })
            .moveDecimalPointLeft(2)
            .moveDecimalPointRight(1)
            .trim(true)
            .toString()).toBe("12.3456 TEST");
    });
    it("Test CoinPretty's toString()", () => {
        const tests = [
            {
                base: new coin_pretty_1.CoinPretty({
                    coinDenom: "ATOM",
                    coinMinimalDenom: "uatom",
                    coinDecimals: 6,
                }, new int_1.Int("12345600")),
                pre: (pretty) => pretty,
                res: "12.345600 ATOM",
            },
            {
                base: new coin_pretty_1.CoinPretty({
                    coinDenom: "ATOM",
                    coinMinimalDenom: "uatom",
                    coinDecimals: 6,
                }, new int_1.Int("12345600")),
                pre: (pretty) => pretty.hideDenom(true),
                res: "12.345600",
            },
            {
                base: new coin_pretty_1.CoinPretty({
                    coinDenom: "ATOM",
                    coinMinimalDenom: "uatom",
                    coinDecimals: 6,
                }, new int_1.Int("12345600")),
                pre: (pretty) => pretty.separator(""),
                res: "12.345600ATOM",
            },
            {
                base: new coin_pretty_1.CoinPretty({
                    coinDenom: "ATOM",
                    coinMinimalDenom: "uatom",
                    coinDecimals: 6,
                }, new int_1.Int("12345600")),
                pre: (pretty) => pretty.maxDecimals(3),
                res: "12.345 ATOM",
            },
            {
                base: new coin_pretty_1.CoinPretty({
                    coinDenom: "ATOM",
                    coinMinimalDenom: "uatom",
                    coinDecimals: 6,
                }, new int_1.Int("12345600")),
                pre: (pretty) => pretty.maxDecimals(3).separator(""),
                res: "12.345ATOM",
            },
            {
                base: new coin_pretty_1.CoinPretty({
                    coinDenom: "ATOM",
                    coinMinimalDenom: "uatom",
                    coinDecimals: 6,
                }, new int_1.Int("12345600")),
                pre: (pretty) => pretty.lowerCase(true),
                res: "12.345600 atom",
            },
            {
                base: new coin_pretty_1.CoinPretty({
                    coinDenom: "AtoM",
                    coinMinimalDenom: "uatom",
                    coinDecimals: 6,
                }, new int_1.Int("12345600")),
                pre: (pretty) => pretty.upperCase(true),
                res: "12.345600 ATOM",
            },
            {
                base: new coin_pretty_1.CoinPretty({
                    coinDenom: "ATOM",
                    coinMinimalDenom: "uatom",
                    coinDecimals: 6,
                }, new int_1.Int("1234560000")),
                pre: (pretty) => pretty,
                res: "1,234.560000 ATOM",
            },
            {
                base: new coin_pretty_1.CoinPretty({
                    coinDenom: "ATOM",
                    coinMinimalDenom: "uatom",
                    coinDecimals: 6,
                }, new int_1.Int("1234560000")),
                pre: (pretty) => pretty.locale(false),
                res: "1234.560000 ATOM",
            },
            {
                base: new coin_pretty_1.CoinPretty({
                    coinDenom: "ATOM",
                    coinMinimalDenom: "uatom",
                    coinDecimals: 6,
                }, new int_1.Int("1234560000")),
                pre: (pretty) => pretty.shrink(true),
                res: coin_utils_1.CoinUtils.shrinkDecimals(new decimal_1.Dec("1234560000").mul(dec_utils_1.DecUtils.getTenExponentN(-6)), 0, 6, true) + " ATOM",
            },
            {
                base: new coin_pretty_1.CoinPretty({
                    coinDenom: "ATOM",
                    coinMinimalDenom: "uatom",
                    coinDecimals: 6,
                }, new int_1.Int("1234560000")),
                pre: (pretty) => pretty.locale(false).shrink(true),
                res: coin_utils_1.CoinUtils.shrinkDecimals(new decimal_1.Dec("1234560000").mul(dec_utils_1.DecUtils.getTenExponentN(-6)), 0, 6, false) + " ATOM",
            },
        ];
        for (const test of tests) {
            expect(test.pre(test.base).toString()).toBe(test.res);
        }
    });
});
//# sourceMappingURL=coin-pretty.spec.js.map