import { schema } from 'prosemirror-schema-basic';
import { addListNodes } from 'prosemirror-schema-list';
import { Node, Schema } from 'prosemirror-model';

var noTag = Node.prototype.tag = Object.create(null);

function flatten(schema, children, f) {
  var result = [], pos = 0, tag = noTag;

  for (var i = 0; i < children.length; i++) {
    var child = children[i];
    if (child.tag && child.tag != Node.prototype.tag) {
      if (tag == noTag) { tag = Object.create(null); }
      for (var id in child.tag)
        { tag[id] = child.tag[id] + (child.flat || child.isText ? 0 : 1) + pos; }
    }

    if (typeof child == "string") {
      var re = /<(\w+)>/g, m = (void 0), at = 0, out = "";
      while (m = re.exec(child)) {
        out += child.slice(at, m.index);
        pos += m.index - at;
        at = m.index + m[0].length;
        if (tag == noTag) { tag = Object.create(null); }
        tag[m[1]] = pos;
      }
      out += child.slice(at);
      pos += child.length - at;
      if (out) { result.push(f(schema.text(out))); }
    } else if (child.flat) {
      for (var j = 0; j < child.flat.length; j++) {
        var node = f(child.flat[j]);
        pos += node.nodeSize;
        result.push(node);
      }
    } else {
      var node$1 = f(child);
      pos += node$1.nodeSize;
      result.push(node$1);
    }
  }
  return {nodes: result, tag: tag}
}

function id(x) { return x }

function takeAttrs(attrs, args) {
  var a0 = args[0];
  if (!args.length || (a0 && (typeof a0 == "string" || a0 instanceof Node || a0.flat)))
    { return attrs }

  args.shift();
  if (!attrs) { return a0 }
  if (!a0) { return attrs }
  var result = {};
  for (var prop in attrs) { result[prop] = attrs[prop]; }
  for (var prop$1 in a0) { result[prop$1] = a0[prop$1]; }
  return result
}

// : (string, ?Object) → (...content: [union<string, Node>]) → Node
// Create a builder function for nodes with content.
function block(type, attrs) {
  var result = function() {
    var args = [], len = arguments.length;
    while ( len-- ) args[ len ] = arguments[ len ];

    var myAttrs = takeAttrs(attrs, args);
    var ref = flatten(type.schema, args, id);
    var nodes = ref.nodes;
    var tag = ref.tag;
    var node = type.create(myAttrs, nodes);
    if (tag != noTag) { node.tag = tag; }
    return node
  };
  if (type.isLeaf) { try { result.flat = [type.create(attrs)]; } catch(_) {} }
  return result
}

// Create a builder function for marks.
function mark(type, attrs) {
  return function() {
    var args = [], len = arguments.length;
    while ( len-- ) args[ len ] = arguments[ len ];

    var mark = type.create(takeAttrs(attrs, args));
    var ref = flatten(type.schema, args, function (n) { return mark.type.isInSet(n.marks) ? n : n.mark(mark.addToSet(n.marks)); });
    var nodes = ref.nodes;
    var tag = ref.tag;
    return {flat: nodes, tag: tag}
  }
}

function builders(schema, names) {
  var result = {schema: schema};
  for (var name in schema.nodes) { result[name] = block(schema.nodes[name], {}); }
  for (var name$1 in schema.marks) { result[name$1] = mark(schema.marks[name$1], {}); }

  if (names) { for (var name$2 in names) {
    var value = names[name$2], typeName = value.nodeType || value.markType || name$2, type = (void 0);
    if (type = schema.nodes[typeName]) { result[name$2] = block(type, value); }
    else if (type = schema.marks[typeName]) { result[name$2] = mark(type, value); }
  } }
  return result
}

var testSchema = new Schema({
  nodes: addListNodes(schema.spec.nodes, "paragraph block*", "block"),
  marks: schema.spec.marks
});

var out = builders(testSchema, {
  p: {nodeType: "paragraph"},
  pre: {nodeType: "code_block"},
  h1: {nodeType: "heading", level: 1},
  h2: {nodeType: "heading", level: 2},
  h3: {nodeType: "heading", level: 3},
  li: {nodeType: "list_item"},
  ul: {nodeType: "bullet_list"},
  ol: {nodeType: "ordered_list"},
  br: {nodeType: "hard_break"},
  img: {nodeType: "image", src: "img.png"},
  hr: {nodeType: "horizontal_rule"},
  a: {markType: "link", href: "foo"},
});

out.eq = function eq(a, b) { return a.eq(b) };
out.builders = builders;

export default out;
//# sourceMappingURL=index.es.js.map
