"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.Signed = void 0;
const nacl = __importStar(require("tweetnacl"));
const utils_1 = require("@spliterati/utils");
const shamir_1 = require("@spliterati/shamir");
var Signed;
(function (Signed) {
    class SignedError extends Error {
    }
    class CryptoError extends SignedError {
    }
    class Shard {
        constructor(keyID, t, n, share) {
            this.keyID = keyID;
            this.t = t;
            this.n = n;
            this.share = share;
            if (keyID.length !== Shard.KEYID_LENGTH) {
                throw new SyntaxError('invalid keyID size');
            }
            if (share.length < 2) {
                throw new SyntaxError('share length too short');
            }
        }
        pack() {
            return Uint8Array.of(...this.keyID, this.t, this.n, ...this.share);
        }
        static unpack(bytes, isSigned = false) {
            if (!bytes) {
                return null;
            }
            const bs = isSigned ? bytes.slice(nacl.sign.signatureLength) : bytes;
            if (bs.length < Shard.KEYID_LENGTH + 2) {
                throw new SyntaxError('share invalid -- too short');
            }
            const slicer = new utils_1.Slicer(bs);
            return new Shard(slicer.next(Shard.KEYID_LENGTH), slicer.next(), slicer.next(), slicer.next('end'));
        }
        metadataEqual(that) {
            return this.n === that.n
                && this.t === that.t
                && utils_1.Uint8ArrayEqual(this.keyID, that.keyID);
        }
    }
    Shard.KEYID_LENGTH = 16;
    Signed.Shard = Shard;
    function generate(t, n, opts = {}) {
        const signingKeys = nacl.sign.keyPair();
        const encryptionKeys = nacl.box.keyPair();
        let keyID;
        if (opts.keyID != null) {
            if (opts.keyID.length !== Shard.KEYID_LENGTH) {
                throw new SyntaxError(`keyID must be ${Shard.KEYID_LENGTH} characters long`);
            }
            keyID = opts.keyID;
        }
        else {
            keyID = nacl.randomBytes(Shard.KEYID_LENGTH);
        }
        const shards = [];
        shamir_1.Shamir.split(encryptionKeys.secretKey, n, t).forEach((share) => {
            shards.push(nacl.sign(new Shard(keyID, t, n, share).pack(), signingKeys.secretKey));
        });
        return {
            signingPublicKey: signingKeys.publicKey,
            encryptionPublicKey: encryptionKeys.publicKey,
            keyID,
            shards,
            signedMessage: opts.message ? nacl.sign(opts.message, signingKeys.secretKey) : null,
        };
    }
    Signed.generate = generate;
    function reconstruct(signingPublicKey, shards) {
        if (shards.length < 2) {
            throw new SyntaxError('2 or more shards are required for reassembly');
        }
        const firstShard = Shard.unpack(nacl.sign.open(shards[0], signingPublicKey));
        if (firstShard == null) {
            throw new SignedError('could not verify share[0]');
        }
        if (shards.length < firstShard.t) {
            throw new SyntaxError(`insufficient shards for reassembly. ${firstShard.t}..${firstShard.n} required.`);
        }
        if (shards.length > firstShard.n) {
            throw new SyntaxError(`more shards than expected. wanted ${firstShard.t}..${firstShard.n}, got ${shards.length}`);
        }
        const shares = [firstShard.share];
        shards.slice(1).forEach((signedShard, index) => {
            const shard = Shard.unpack(nacl.sign.open(signedShard, signingPublicKey));
            if (shard == null) {
                throw new CryptoError(`could not verify shard[${1 + index}]`);
            }
            if (!firstShard.metadataEqual(shard)) {
                throw new SignedError(`metadata mismatch for shard[${1 + index}]`);
            }
            shares.push(shard.share);
        });
        return {
            keyID: firstShard.keyID,
            encryptionKeyPair: nacl.box.keyPair.fromSecretKey(shamir_1.Shamir.combine(shares)),
        };
    }
    Signed.reconstruct = reconstruct;
})(Signed = exports.Signed || (exports.Signed = {}));
//# sourceMappingURL=signed.js.map