// Copyright (C) 2016 Dmitry Chestnykh
// MIT License. See LICENSE file for details.

import { encode, decode } from "@stablelib/hex";
import { concat } from "@stablelib/bytes";
import { oneTimeAuth, Poly1305 } from "./poly1305";

const testVectors = [
        {
            data: "48656C6C6F20776F726C6421",
            key: "746869732069732033322D62797465206B657920666F7220506F6C7931333035",
            mac: "A6F745008F81C916A20DCC74EEF2B2F0"
        },
        {
            data: "0000000000000000000000000000000000000000000000000000000000000000",
            key: "746869732069732033322D62797465206B657920666F7220506F6C7931333035",
            mac: "49EC78090E481EC6C26B33B91CCC0307"
        },
        {
            data: new Array(2007 + 1).join("00"),
            key: "746869732069732033322D62797465206B657920666F7220506F6C7931333035",
            mac: "DA84BCAB02676C38CDB015604274C2AA"
        },
        {
            data: new Array(2007 + 1).join("00"),
            key: "0000000000000000000000000000000000000000000000000000000000000000",
            mac: "00000000000000000000000000000000"
        },
        {
            data: "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F" +
            "202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F" +
            "404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F" +
            "606162636465666768696A6B6C6D6E6F",
            key: "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F",
            mac: "B98AD2F7EC2AE52AFBA1E66FE9133817"
        },
        /*
     * RFC7539
     */
        {
            data: "43727970746F6772617068696320466F72756D2052657365617263682047726F7570",
            key: "85D6BE7857556D337F4452FE42D506A80103808AFB0DB2FD4ABFF6AF4149F51B",
            mac: "A8061DC1305136C6C22B8BAF0C0127A9"
        },
        /*
         * Test vectors from "The Poly1305-AES message-authentication Code"
         */
        {
            data: "F3F6",
            key: "851FC40C3467AC0BE05CC20404F3F700580B3B0F9447BB1E69D095B5928B6DBC",
            mac: "F4C633C3044FC145F84F335CB81953DE"
        },
        {
            data: "",
            key: "A0F3080000F46400D0C7E9076C834403DD3FAB2251F11AC759F0887129CC2EE7",
            mac: "DD3FAB2251F11AC759F0887129CC2EE7"
        },
        {
            data: "663CEA190FFB83D89593F3F476B6BC24D7E679107EA26ADB8CAF6652D0656136",
            key: "48443D0BB0D21109C89A100B5CE2C20883149C69B561DD88298A1798B10716EF",
            mac: "0EE1C16BB73F0F4FD19881753C01CDBE"
        },
        {
            data: "AB0812724A7F1E342742CBED374D94D136C6B8795D45B3819830F2C04491FAF0" +
            "990C62E48B8018B2C3E4A0FA3134CB67FA83E158C994D961C4CB21095C1BF9",
            key: "12976A08C4426D0CE8A82407C4F4820780F8C20AA71202D1E29179CBCB555A57",
            mac: "5154AD0D2CB26E01274FC51148491F1B"
        },
        /*
         * Self-generated
         */
        {
            data: "AB0812724A7F1E342742CBED374D94D136C6B8795D45B3819830F2C04491FAF0" +
            "990C62E48B8018B2C3E4A0FA3134CB67FA83E158C994D961C4CB21095C1BF9AF",
            key: "12976A08C4426D0CE8A82407C4F4820780F8C20AA71202D1E29179CBCB555A57",
            mac: "812059A5DA198637CAC7C4A631BEE466"
        },
        {
            data: "AB0812724A7F1E342742CBED374D94D136C6B8795D45B3819830F2C04491FAF0" +
            "990C62E48B8018B2C3E4A0FA3134CB67",
            key: "12976A08C4426D0CE8A82407C4F4820780F8C20AA71202D1E29179CBCB555A57",
            mac: "5B88D7F6228B11E2E28579A5C0C1F761"
        },
        {
            data: "AB0812724A7F1E342742CBED374D94D136C6B8795D45B3819830F2C04491FAF0" +
            "990C62E48B8018B2C3E4A0FA3134CB67FA83E158C994D961C4CB21095C1BF9AF" +
            "663CEA190FFB83D89593F3F476B6BC24D7E679107EA26ADB8CAF6652D0656136",
            key: "12976A08C4426D0CE8A82407C4F4820780F8C20AA71202D1E29179CBCB555A57",
            mac: "BBB613B2B6D753BA07395B916AAECE15"
        },
        {
            data: "AB0812724A7F1E342742CBED374D94D136C6B8795D45B3819830F2C04491FAF0" +
            "990C62E48B8018B2C3E4A0FA3134CB67FA83E158C994D961C4CB21095C1BF9AF" +
            "48443D0BB0D21109C89A100B5CE2C20883149C69B561DD88298A1798B10716EF" +
            "663CEA190FFB83D89593F3F476B6BC24",
            key: "12976A08C4426D0CE8A82407C4F4820780F8C20AA71202D1E29179CBCB555A57",
            mac: "C794D7057D1778C4BBEE0A39B3D97342"
        },
        {
            data: "AB0812724A7F1E342742CBED374D94D136C6B8795D45B3819830F2C04491FAF0" +
            "990C62E48B8018B2C3E4A0FA3134CB67FA83E158C994D961C4CB21095C1BF9AF" +
            "48443D0BB0D21109C89A100B5CE2C20883149C69B561DD88298A1798B10716EF" +
            "663CEA190FFB83D89593F3F476B6BC24D7E679107EA26ADB8CAF6652D0656136",
            key: "12976A08C4426D0CE8A82407C4F4820780F8C20AA71202D1E29179CBCB555A57",
            mac: "FFBCB9B371423152D7FCA5AD042FBAA9"
        },
        {
            data: "AB0812724A7F1E342742CBED374D94D136C6B8795D45B3819830F2C04491FAF0" +
            "990C62E48B8018B2C3E4A0FA3134CB67FA83E158C994D961C4CB21095C1BF9AF" +
            "48443D0BB0D21109C89A100B5CE2C20883149C69B561DD88298A1798B10716EF" +
            "663CEA190FFB83D89593F3F476B6BC24D7E679107EA26ADB8CAF6652D0656136" +
            "812059A5DA198637CAC7C4A631BEE466",
            key: "12976A08C4426D0CE8A82407C4F4820780F8C20AA71202D1E29179CBCB555A57",
            mac: "069ED6B8EF0F207B3E243BB1019FE632"
        },
        {
            data: "AB0812724A7F1E342742CBED374D94D136C6B8795D45B3819830F2C04491FAF0" +
            "990C62E48B8018B2C3E4A0FA3134CB67FA83E158C994D961C4CB21095C1BF9AF" +
            "48443D0BB0D21109C89A100B5CE2C20883149C69B561DD88298A1798B10716EF" +
            "663CEA190FFB83D89593F3F476B6BC24D7E679107EA26ADB8CAF6652D0656136" +
            "812059A5DA198637CAC7C4A631BEE4665B88D7F6228B11E2E28579A5C0C1F761",
            key: "12976A08C4426D0CE8A82407C4F4820780F8C20AA71202D1E29179CBCB555A57",
            mac: "CCA339D9A45FA2368C2C68B3A4179133"
        },
        {
            data: "AB0812724A7F1E342742CBED374D94D136C6B8795D45B3819830F2C04491FAF0" +
            "990C62E48B8018B2C3E4A0FA3134CB67FA83E158C994D961C4CB21095C1BF9AF" +
            "48443D0BB0D21109C89A100B5CE2C20883149C69B561DD88298A1798B10716EF" +
            "663CEA190FFB83D89593F3F476B6BC24D7E679107EA26ADB8CAF6652D0656136" +
            "812059A5DA198637CAC7C4A631BEE4665B88D7F6228B11E2E28579A5C0C1F761" +
            "AB0812724A7F1E342742CBED374D94D136C6B8795D45B3819830F2C04491FAF0" +
            "990C62E48B8018B2C3E4A0FA3134CB67FA83E158C994D961C4CB21095C1BF9AF" +
            "48443D0BB0D21109C89A100B5CE2C20883149C69B561DD88298A1798B10716EF" +
            "663CEA190FFB83D89593F3F476B6BC24D7E679107EA26ADB8CAF6652D0656136",
            key: "12976A08C4426D0CE8A82407C4F4820780F8C20AA71202D1E29179CBCB555A57",
            mac: "53F6E828A2F0FE0EE815BF0BD5841A34"
        },
        {
            data: "AB0812724A7F1E342742CBED374D94D136C6B8795D45B3819830F2C04491FAF0" +
            "990C62E48B8018B2C3E4A0FA3134CB67FA83E158C994D961C4CB21095C1BF9AF" +
            "48443D0BB0D21109C89A100B5CE2C20883149C69B561DD88298A1798B10716EF" +
            "663CEA190FFB83D89593F3F476B6BC24D7E679107EA26ADB8CAF6652D0656136" +
            "812059A5DA198637CAC7C4A631BEE4665B88D7F6228B11E2E28579A5C0C1F761" +
            "AB0812724A7F1E342742CBED374D94D136C6B8795D45B3819830F2C04491FAF0" +
            "990C62E48B8018B2C3E4A0FA3134CB67FA83E158C994D961C4CB21095C1BF9AF" +
            "48443D0BB0D21109C89A100B5CE2C20883149C69B561DD88298A1798B10716EF" +
            "663CEA190FFB83D89593F3F476B6BC24D7E679107EA26ADB8CAF6652D0656136" +
            "812059A5DA198637CAC7C4A631BEE4665B88D7F6228B11E2E28579A5C0C1F761",
            key: "12976A08C4426D0CE8A82407C4F4820780F8C20AA71202D1E29179CBCB555A57",
            mac: "B846D44E9BBD53CEDFFBFBB6B7FA4933"
        },
        {
            /*
             * poly1305_ieee754.c failed this in final stage
             */
            data: "842364E156336C0998B933A6237726180D9E3FDCBDE4CD5D17080FC3BEB49614" +
            "D7122C037463FF104D73F19C12704628D417C4C54A3FE30D3C3D7714382D43B0" +
            "382A50A5DEE54BE844B076E8DF88201A1CD43B90EB21643FA96F39B518AA8340" +
            "C942FF3C31BAF7C9BDBF0F31AE3FA096BF8C63030609829FE72E179824890BC8" +
            "E08C315C1CCE2A83144DBBFF09F74E3EFC770B54D0984A8F19B14719E6363564" +
            "1D6B1EEDF63EFBF080E1783D32445412114C20DE0B837A0DFA33D6B82825FFF4" +
            "4C9A70EA54CE47F07DF698E6B03323B53079364A5FC3E9DD034392BDDE86DCCD" +
            "DA94321C5E44060489336CB65BF3989C36F7282C2F5D2B882C171E74",
            key: "95D5C005503E510D8CD0AA072C4A4D066EABC52D11653DF47FBF63AB198BCC26",
            mac: "F248312E578D9D58F8B7BB4D19105431"
        },
        /*
         * AVX2 in poly1305-x86.pl failed this with 176+32 split
         */
        {
            data: "248AC31085B6C2ADAAA38259A0D7192C5C35D1BB4EF39AD94C38D1C82479E2DD" +
            "2159A077024B0589BC8A20101B506F0A1AD0BBAB76E83A83F1B94BE6BEAE74E8" +
            "74CAB692C5963A75436B776121EC9F62399A3E66B2D22707DAE81933B6277F3C" +
            "8516BCBE26DBBD86F373103D7CF4CAD1888C952118FBFBD0D7B4BEDC4AE4936A" +
            "FF91157E7AA47C54442EA78D6AC251D324A0FBE49D89CC3521B66D16E9C66A37" +
            "09894E4EB0A4EEDC4AE19468E66B81F2" +
            "71351B1D921EA551047ABCC6B87A901FDE7DB79FA1818C11336DBC07244A40EB",
            key: "000102030405060708090A0B0C0D0E0F00000000000000000000000000000000",
            mac: "BC939BC5281480FA99C6D68C258EC42F"
        },
        /*
         * test vectors from Google
         */
        {
            data: "",
            key: "C8AFAAC331EE372CD6082DE134943B174710130E9F6FEA8D72293850A667D86C",
            mac: "4710130E9F6FEA8D72293850A667D86C",
        },
        {
            data: "48656C6C6F20776F726C6421",
            key: "746869732069732033322D62797465206B657920666F7220506F6C7931333035",
            mac: "A6F745008F81C916A20DCC74EEF2B2F0"
        },
        {
            data: "0000000000000000000000000000000000000000000000000000000000000000",
            key: "746869732069732033322D62797465206B657920666F7220506F6C7931333035",
            mac: "49EC78090E481EC6C26B33B91CCC0307"
        },
        /*
         * test vectors from Andrew Moon
         */
        {   /* nacl */
            data: "8E993B9F48681273C29650BA32FC76CE48332EA7164D96A4476FB8C531A1186A" +
            "C0DFC17C98DCE87B4DA7F011EC48C97271D2C20F9B928FE2270D6FB863D51738" +
            "B48EEEE314A7CC8AB932164548E526AE90224368517ACFEABD6BB3732BC0E9DA" +
            "99832B61CA01B6DE56244A9E88D5F9B37973F622A43D14A6599B1F654CB45A74" +
            "E355A5",
            key: "EEA6A7251C1E72916D11C2CB214D3C252539121D8E234E652D651FA4C8CFF880",
            mac: "F3FFC7703F9400E52A7DFB4B3D3305D9"
        },
    {   /* wrap 2^130-5 */
        data: "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF",
        key: "0200000000000000000000000000000000000000000000000000000000000000",
        mac: "03000000000000000000000000000000"
    },
    {   /* wrap 2^128 */
        data: "02000000000000000000000000000000",
        key: "02000000000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF",
        mac: "03000000000000000000000000000000"
    },
    {   /* limb carry */
        data: "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF0FFFFFFFFFFFFFFFFFFFFFFFFFFFFFF" +
        "11000000000000000000000000000000",
        key: "0100000000000000000000000000000000000000000000000000000000000000",
        mac: "05000000000000000000000000000000"

    },
    {   /* 2^130-5 */
        data: "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFBFEFEFEFEFEFEFEFEFEFEFEFEFEFEFE" +
        "01010101010101010101010101010101",
        key: "0100000000000000000000000000000000000000000000000000000000000000",
        mac: "00000000000000000000000000000000"
    },
    {   /* 2^130-6 */
        data: "FDFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF",
        key: "0200000000000000000000000000000000000000000000000000000000000000",
        mac: "FAFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF"
    },
    {   /* 5*H+L reduction intermediate */
        data: "E33594D7505E43B900000000000000003394D7505E4379CD0100000000000000" +
        "0000000000000000000000000000000001000000000000000000000000000000",
        key: "0100000000000000040000000000000000000000000000000000000000000000",
        mac: "14000000000000005500000000000000"
    },
    {   /* 5*H+L REDUCTION FINAL */
        data: "E33594D7505E43B900000000000000003394D7505E4379CD0100000000000000" +
        "00000000000000000000000000000000",
        key: "0100000000000000040000000000000000000000000000000000000000000000",
        mac: "13000000000000000000000000000000"
    }
];

describe("poly1305", () => {
    it("should produce correct values", () => {
        testVectors.forEach(v => {
            const mac = oneTimeAuth(decode(v.key), decode(v.data));
            expect(encode(mac)).toBe(v.mac);
        });
    });

    it('multiple update should produce the same result as a single update', () => {
        const key = new Uint8Array(32);
        for (let i = 0; i < key.length; i++) {
            key[i] = i;
        }
        const data = new Uint8Array(4 + 64 + 169);
        for (let i = 0; i < data.length; i++) {
            data[i] = i;
        }
        const d1 = data.subarray(0, 4);
        const d2 = data.subarray(4, 4 + 64);
        const d3 = data.subarray(4 + 64);

        expect(encode(concat(d1, d2, d3))).toBe(encode(data));

        const p1 = new Poly1305(key);
        p1.update(data);
        const r1 = p1.digest();

        const p2 = new Poly1305(key);
        p2.update(d1);
        p2.update(d2);
        p2.update(d3);
        const r2 = p2.digest();
        expect(encode(r1)).toBe(encode(r2));
    });
});
