"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.copyHTMLFiles = exports.copyFunctionLibs = exports.writePrerenderConfig = exports.writeVCConfig = exports.writeHandler = void 0;
const path_1 = require("path");
const build_utils_1 = require("@vercel/build-utils");
const esbuild_1 = require("esbuild");
const fs_extra_1 = require("fs-extra");
const writeHandler = async ({ outDir, handlerFile, }) => {
    const { major } = await build_utils_1.getNodeVersion(process.cwd());
    try {
        return await esbuild_1.build({
            entryPoints: [handlerFile],
            loader: { '.ts': 'ts' },
            outfile: path_1.join(outDir, './index.js'),
            format: 'cjs',
            target: `node${major}`,
            platform: 'node',
            bundle: true,
            minify: true,
            define: {
                'process.env.NODE_ENV': "'production'",
            },
        });
    }
    catch (e) {
        console.error('Failed to build lambda handler', e.message);
    }
};
exports.writeHandler = writeHandler;
const writeVCConfig = async ({ functionDir, handler = 'index.js', }) => {
    const { runtime } = await build_utils_1.getNodeVersion(process.cwd());
    const config = {
        runtime,
        handler,
        launcherType: 'Nodejs',
        shouldAddHelpers: true,
    };
    return fs_extra_1.writeJson(`${functionDir}/.vc-config.json`, config);
};
exports.writeVCConfig = writeVCConfig;
const writePrerenderConfig = (outputPath, group) => {
    const config = {
        group,
        expiration: 600, // 10 minutes TODO: make this configurable?
    };
    fs_extra_1.ensureFileSync(outputPath);
    return fs_extra_1.writeFileSync(outputPath, JSON.stringify(config));
};
exports.writePrerenderConfig = writePrerenderConfig;
async function copyFunctionLibs({ functionDir, }) {
    /* Copies the required libs for Serverless Functions from .cache to the <name>.func folder */
    await Promise.allSettled([
        {
            src: path_1.join('.cache', 'query-engine'),
            dest: path_1.join(functionDir, '.cache', 'query-engine'),
        },
        {
            src: path_1.join('.cache', 'page-ssr'),
            dest: path_1.join(functionDir, '.cache', 'page-ssr'),
        },
        // {
        //   src: join(functionDir, '.cache', 'query-engine', 'assets'),
        //   dest: join(functionDir, 'assets'),
        // },
        {
            src: path_1.join('.cache', 'data', 'datastore'),
            dest: path_1.join(functionDir, '.cache', 'data', 'datastore'),
        },
        {
            src: path_1.join('.cache', 'caches'),
            dest: path_1.join(functionDir, '.cache', 'caches'),
        },
    ].map(({ src, dest }) => fs_extra_1.copy(src, dest)));
}
exports.copyFunctionLibs = copyFunctionLibs;
async function copyHTMLFiles({ functionDir }) {
    /* If available, copies the 404.html and 500.html files to the <name>.func/lib folder */
    for (const htmlFile of ['404', '500']) {
        if (await fs_extra_1.pathExists(path_1.join('public', `${htmlFile}.html`))) {
            try {
                await fs_extra_1.copyFile(path_1.join('public', `${htmlFile}.html`), path_1.join(functionDir, `${htmlFile}.html`));
            }
            catch (e) {
                console.error('Failed to copy HTML files', e.message);
                process.exit(1);
            }
        }
    }
}
exports.copyHTMLFiles = copyHTMLFiles;
//# sourceMappingURL=build.js.map