"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateVercelBuildOutputAPI3Output = void 0;
const routing_utils_1 = require("@vercel/routing-utils");
const fs_extra_1 = require("fs-extra");
const schemas_1 = require("./schemas");
const functions_1 = require("./helpers/functions");
const static_1 = require("./helpers/static");
async function generateVercelBuildOutputAPI3Output({ gatsbyStoreState, }) {
    const state = {
        pages: Array.from(gatsbyStoreState.pages.entries()),
        redirects: gatsbyStoreState.redirects,
        functions: gatsbyStoreState.functions,
        config: gatsbyStoreState.config,
    };
    if (schemas_1.validateGatsbyState.Check(state)) {
        console.log('▲ Creating Vercel build output');
        const { pages, redirects, functions, config: gatsbyConfig } = state;
        const { pathPrefix = '' } = gatsbyConfig;
        const ssrRoutes = pages
            .map(p => p[1])
            .filter(page => page.mode === 'SSR' || page.mode === 'DSG');
        const ops = [];
        if (functions.length > 0) {
            ops.push((0, functions_1.createAPIRoutes)(functions, pathPrefix));
        }
        if (ssrRoutes.length > 0) {
            ops.push((0, functions_1.createServerlessFunctions)(ssrRoutes, pathPrefix));
        }
        await Promise.all(ops);
        // "static" directory needs to happen last since it moves "public"
        await (0, static_1.createStaticDir)(pathPrefix);
        let trailingSlash = undefined;
        if (gatsbyConfig.trailingSlash === 'always') {
            trailingSlash = true;
        }
        else if (gatsbyConfig.trailingSlash === 'never') {
            trailingSlash = false;
        }
        const { routes } = (0, routing_utils_1.getTransformedRoutes)({
            trailingSlash,
            redirects: redirects.map(({ fromPath, toPath, isPermanent }) => ({
                source: fromPath,
                destination: toPath,
                permanent: isPermanent,
            })),
        });
        const config = {
            version: 3,
            routes: routes || undefined,
        };
        await (0, fs_extra_1.writeJson)('.vercel/output/config.json', config);
        console.log('Vercel output has been generated');
    }
    else {
        const errors = [...schemas_1.validateGatsbyState.Errors(state)];
        throw new Error(`Gatsby state validation failed:\n${errors
            .map(err => `  - ${err.message}, got ${typeof err.value} (${JSON.stringify(err.value)}) at path "${err.path}"\n`)
            .join('')}Please check your Gatsby configuration files, or file an issue at https://vercel.com/help#issues`);
    }
}
exports.generateVercelBuildOutputAPI3Output = generateVercelBuildOutputAPI3Output;
//# sourceMappingURL=index.js.map