"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.serializeBody = exports.validateConfiguredRuntime = exports.isEdgeRuntime = exports.EdgeRuntimes = exports.logError = exports.entrypointToOutputPath = exports.getRegExpFromMatchers = void 0;
const build_utils_1 = require("@vercel/build-utils");
const path_to_regexp_1 = require("path-to-regexp");
const path_1 = require("path");
function getRegExpFromMatchers(matcherOrMatchers) {
    if (!matcherOrMatchers) {
        return '^/.*$';
    }
    const matchers = Array.isArray(matcherOrMatchers)
        ? matcherOrMatchers
        : [matcherOrMatchers];
    const regExps = matchers.flatMap(getRegExpFromMatcher).join('|');
    return regExps;
}
exports.getRegExpFromMatchers = getRegExpFromMatchers;
function getRegExpFromMatcher(matcher, index, allMatchers) {
    if (typeof matcher !== 'string') {
        throw new Error("Middleware's `config.matcher` must be a path matcher (string) or an array of path matchers (string[])");
    }
    if (!matcher.startsWith('/')) {
        throw new Error(`Middleware's \`config.matcher\` values must start with "/". Received: ${matcher}`);
    }
    const regExps = [(0, path_to_regexp_1.pathToRegexp)(matcher).source];
    if (matcher === '/' && !allMatchers.includes('/index')) {
        regExps.push((0, path_to_regexp_1.pathToRegexp)('/index').source);
    }
    return regExps;
}
/**
 * If `zeroConfig`:
 *   "api/foo.js" -> "api/foo.js"
 *   "api/foo.ts" -> "api/foo.ts"
 *
 * If *NOT* `zeroConfig`:
 *   "api/foo.js" -> "api/foo"
 *   "api/foo.ts" -> "api/foo"
 */
function entrypointToOutputPath(entrypoint, zeroConfig) {
    if (zeroConfig) {
        const ext = (0, path_1.extname)(entrypoint);
        return entrypoint.slice(0, entrypoint.length - ext.length);
    }
    return entrypoint;
}
exports.entrypointToOutputPath = entrypointToOutputPath;
function logError(error) {
    let message = error.message;
    if (!message.startsWith('Error:')) {
        message = `Error: ${message}`;
    }
    console.error(message);
    if (error.stack) {
        // only show the stack trace if debug is enabled
        // because it points to internals, not user code
        const errorPrefixLength = 'Error: '.length;
        const errorMessageLength = errorPrefixLength + error.message.length;
        (0, build_utils_1.debug)(error.stack.substring(errorMessageLength + 1));
    }
}
exports.logError = logError;
var EdgeRuntimes;
(function (EdgeRuntimes) {
    EdgeRuntimes["Edge"] = "edge";
    EdgeRuntimes["ExperimentalEdge"] = "experimental-edge";
})(EdgeRuntimes = exports.EdgeRuntimes || (exports.EdgeRuntimes = {}));
function isEdgeRuntime(runtime) {
    return (runtime !== undefined &&
        Object.values(EdgeRuntimes).includes(runtime));
}
exports.isEdgeRuntime = isEdgeRuntime;
const ALLOWED_RUNTIMES = Object.values(EdgeRuntimes);
function validateConfiguredRuntime(runtime, entrypoint) {
    if (runtime) {
        if (runtime === 'nodejs') {
            throw new Error(`${entrypoint}: \`config.runtime: "nodejs"\` semantics will evolve soon. Please remove the \`runtime\` key to keep the existing behavior.`);
        }
        if (!ALLOWED_RUNTIMES.includes(runtime)) {
            throw new Error(`${entrypoint}: unsupported "runtime" value in \`config\`: ${JSON.stringify(runtime)} (must be one of: ${JSON.stringify(ALLOWED_RUNTIMES)}). Learn more: https://vercel.link/creating-edge-functions`);
        }
    }
}
exports.validateConfiguredRuntime = validateConfiguredRuntime;
async function serializeBody(request) {
    return request.method !== 'GET' && request.method !== 'HEAD'
        ? await (0, build_utils_1.streamToBuffer)(request)
        : undefined;
}
exports.serializeBody = serializeBody;
