"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.build = void 0;
const ts_morph_1 = require("ts-morph");
const fs_1 = require("fs");
const path_1 = require("path");
const build_utils_1 = require("@vercel/build-utils");
const static_config_1 = require("@vercel/static-config");
const nft_1 = require("@vercel/nft");
const utils_1 = require("./utils");
const remixBuilderPkg = JSON.parse((0, fs_1.readFileSync)((0, path_1.join)(__dirname, '../package.json'), 'utf8'));
const remixRunDevForkVersion = remixBuilderPkg.devDependencies['@remix-run/dev'];
const DEFAULTS_PATH = (0, path_1.join)(__dirname, '../defaults');
const edgeServerSrcPromise = fs_1.promises.readFile((0, path_1.join)(DEFAULTS_PATH, 'server-edge.mjs'), 'utf-8');
const nodeServerSrcPromise = fs_1.promises.readFile((0, path_1.join)(DEFAULTS_PATH, 'server-node.mjs'), 'utf-8');
// Minimum supported version of the `@vercel/remix` package
const VERCEL_REMIX_MIN_VERSION = '1.10.0';
// Minimum supported version of the `@vercel/remix-run-dev` forked compiler
const REMIX_RUN_DEV_MIN_VERSION = '1.15.0';
// Maximum version of `@vercel/remix-run-dev` fork
// (and also `@vercel/remix` since they get published at the same time)
const REMIX_RUN_DEV_MAX_VERSION = remixRunDevForkVersion.slice(remixRunDevForkVersion.lastIndexOf('@') + 1);
const build = async ({ entrypoint, files, workPath, repoRootPath, config, meta = {}, }) => {
    const { installCommand, buildCommand } = config;
    await (0, build_utils_1.download)(files, workPath, meta);
    const mountpoint = (0, path_1.dirname)(entrypoint);
    const entrypointFsDirname = (0, path_1.join)(workPath, mountpoint);
    // Run "Install Command"
    const nodeVersion = await (0, build_utils_1.getNodeVersion)(entrypointFsDirname, undefined, config, meta);
    const { cliType, packageJsonPath, lockfileVersion } = await (0, build_utils_1.scanParentDirs)(entrypointFsDirname);
    if (!packageJsonPath) {
        throw new Error('Failed to locate `package.json` file in your project');
    }
    const pkgRaw = await fs_1.promises.readFile(packageJsonPath, 'utf8');
    const pkg = JSON.parse(pkgRaw);
    const spawnOpts = (0, build_utils_1.getSpawnOptions)(meta, nodeVersion);
    if (!spawnOpts.env) {
        spawnOpts.env = {};
    }
    spawnOpts.env = (0, build_utils_1.getEnvForPackageManager)({
        cliType,
        lockfileVersion,
        nodeVersion,
        env: spawnOpts.env,
    });
    if (typeof installCommand === 'string') {
        if (installCommand.trim()) {
            console.log(`Running "install" command: \`${installCommand}\`...`);
            await (0, build_utils_1.execCommand)(installCommand, {
                ...spawnOpts,
                cwd: entrypointFsDirname,
            });
        }
        else {
            console.log(`Skipping "install" command...`);
        }
    }
    else {
        await (0, build_utils_1.runNpmInstall)(entrypointFsDirname, [], spawnOpts, meta, nodeVersion);
    }
    // Determine the version of Remix based on the `@remix-run/dev`
    // package version.
    const remixRunDevPath = await (0, utils_1.ensureResolvable)(entrypointFsDirname, repoRootPath, '@remix-run/dev');
    const remixRunDevPkg = JSON.parse((0, fs_1.readFileSync)((0, path_1.join)(remixRunDevPath, 'package.json'), 'utf8'));
    const remixVersion = remixRunDevPkg.version;
    const remixConfig = await (0, utils_1.chdirAndReadConfig)(remixRunDevPath, entrypointFsDirname, packageJsonPath);
    const { serverEntryPoint, appDirectory } = remixConfig;
    const remixRoutes = Object.values(remixConfig.routes);
    const depsToAdd = [];
    if (remixRunDevPkg.name !== '@vercel/remix-run-dev') {
        const remixDevForkVersion = (0, utils_1.resolveSemverMinMax)(REMIX_RUN_DEV_MIN_VERSION, REMIX_RUN_DEV_MAX_VERSION, remixVersion);
        depsToAdd.push(`@remix-run/dev@npm:@vercel/remix-run-dev@${remixDevForkVersion}`);
    }
    // `app/entry.server.tsx` and `app/entry.client.tsx` are optional in Remix,
    // so if either of those files are missing then add our own versions.
    const userEntryServerFile = (0, utils_1.findEntry)(appDirectory, 'entry.server');
    if (!userEntryServerFile) {
        await fs_1.promises.copyFile((0, path_1.join)(DEFAULTS_PATH, 'entry.server.jsx'), (0, path_1.join)(appDirectory, 'entry.server.jsx'));
        if (!pkg.dependencies['@vercel/remix']) {
            // Dependency version resolution logic
            // 1. Users app is on 1.9.0 -> we install the 1.10.0 (minimum) version of `@vercel/remix`.
            // 2. Users app is on 1.11.0 (a version greater than 1.10.0 and less than the known max
            //    published version) -> we install the (matching) 1.11.0 version of `@vercel/remix`.
            // 3. Users app is on something greater than our latest version of the fork -> we install
            //    the latest known published version of `@vercel/remix`.
            const vercelRemixVersion = (0, utils_1.resolveSemverMinMax)(VERCEL_REMIX_MIN_VERSION, REMIX_RUN_DEV_MAX_VERSION, remixVersion);
            depsToAdd.push(`@vercel/remix@${vercelRemixVersion}`);
        }
    }
    if (depsToAdd.length) {
        await (0, utils_1.addDependencies)(cliType, depsToAdd, {
            ...spawnOpts,
            cwd: entrypointFsDirname,
        });
    }
    const userEntryClientFile = (0, utils_1.findEntry)(remixConfig.appDirectory, 'entry.client');
    if (!userEntryClientFile) {
        await fs_1.promises.copyFile((0, path_1.join)(DEFAULTS_PATH, 'entry.client.react.jsx'), (0, path_1.join)(appDirectory, 'entry.client.jsx'));
    }
    let remixConfigWrapped = false;
    const remixConfigPath = (0, utils_1.findConfig)(entrypointFsDirname, 'remix.config');
    const renamedRemixConfigPath = remixConfigPath
        ? `${remixConfigPath}.original${(0, path_1.extname)(remixConfigPath)}`
        : undefined;
    // These get populated inside the try/catch below
    let serverBundles;
    const serverBundlesMap = new Map();
    const resolvedConfigsMap = new Map();
    try {
        // Read the `export const config` (if any) for each route
        const project = new ts_morph_1.Project();
        const staticConfigsMap = new Map();
        for (const route of remixRoutes) {
            const routePath = (0, path_1.join)(remixConfig.appDirectory, route.file);
            const staticConfig = (0, static_config_1.getConfig)(project, routePath);
            staticConfigsMap.set(route, staticConfig);
        }
        for (const route of remixRoutes) {
            const config = (0, utils_1.getResolvedRouteConfig)(route, remixConfig.routes, staticConfigsMap);
            resolvedConfigsMap.set(route, config);
        }
        // Figure out which routes belong to which server bundles
        // based on having common static config properties
        for (const route of remixRoutes) {
            if ((0, utils_1.isLayoutRoute)(route.id, remixRoutes))
                continue;
            const config = resolvedConfigsMap.get(route);
            if (!config) {
                throw new Error(`Expected resolved config for "${route.id}"`);
            }
            const hash = (0, utils_1.calculateRouteConfigHash)(config);
            let routesForHash = serverBundlesMap.get(hash);
            if (!Array.isArray(routesForHash)) {
                routesForHash = [];
                serverBundlesMap.set(hash, routesForHash);
            }
            routesForHash.push(route);
        }
        serverBundles = Array.from(serverBundlesMap.entries()).map(([hash, routes]) => {
            const runtime = resolvedConfigsMap.get(routes[0])?.runtime ?? 'nodejs';
            return {
                serverBuildPath: `build/build-${runtime}-${hash}.js`,
                routes: routes.map(r => r.id),
            };
        });
        // We need to patch the `remix.config.js` file to force some values necessary
        // for a build that works on either Node.js or the Edge runtime
        if (remixConfigPath && renamedRemixConfigPath) {
            await fs_1.promises.rename(remixConfigPath, renamedRemixConfigPath);
            let patchedConfig;
            // Figure out if the `remix.config` file is using ESM syntax
            if ((0, utils_1.isESM)(renamedRemixConfigPath)) {
                patchedConfig = `import config from './${(0, path_1.basename)(renamedRemixConfigPath)}';
config.serverBuildTarget = undefined;
config.serverModuleFormat = 'cjs';
config.serverPlatform = 'node';
config.serverBuildPath = undefined;
config.serverBundles = ${JSON.stringify(serverBundles)};
export default config;`;
            }
            else {
                patchedConfig = `const config = require('./${(0, path_1.basename)(renamedRemixConfigPath)}');
config.serverBuildTarget = undefined;
config.serverModuleFormat = 'cjs';
config.serverPlatform = 'node';
config.serverBuildPath = undefined;
config.serverBundles = ${JSON.stringify(serverBundles)};
module.exports = config;`;
            }
            await fs_1.promises.writeFile(remixConfigPath, patchedConfig);
            remixConfigWrapped = true;
        }
        // Make `remix build` output production mode
        spawnOpts.env.NODE_ENV = 'production';
        // Run "Build Command"
        if (buildCommand) {
            (0, build_utils_1.debug)(`Executing build command "${buildCommand}"`);
            await (0, build_utils_1.execCommand)(buildCommand, {
                ...spawnOpts,
                cwd: entrypointFsDirname,
            });
        }
        else {
            if (hasScript('vercel-build', pkg)) {
                (0, build_utils_1.debug)(`Executing "yarn vercel-build"`);
                await (0, build_utils_1.runPackageJsonScript)(entrypointFsDirname, 'vercel-build', spawnOpts);
            }
            else if (hasScript('build', pkg)) {
                (0, build_utils_1.debug)(`Executing "yarn build"`);
                await (0, build_utils_1.runPackageJsonScript)(entrypointFsDirname, 'build', spawnOpts);
            }
            else {
                await (0, build_utils_1.execCommand)('remix build', {
                    ...spawnOpts,
                    cwd: entrypointFsDirname,
                });
            }
        }
    }
    finally {
        // Clean up our patched `remix.config.js` to be polite
        if (remixConfigWrapped && remixConfigPath && renamedRemixConfigPath) {
            await fs_1.promises.rename(renamedRemixConfigPath, remixConfigPath);
        }
    }
    // This needs to happen before we run NFT to create the Node/Edge functions
    await Promise.all([
        (0, utils_1.ensureResolvable)(entrypointFsDirname, repoRootPath, '@remix-run/server-runtime'),
        (0, utils_1.ensureResolvable)(entrypointFsDirname, repoRootPath, '@remix-run/node'),
    ]);
    const [staticFiles, ...functions] = await Promise.all([
        (0, build_utils_1.glob)('**', (0, path_1.join)(entrypointFsDirname, 'public')),
        ...serverBundles.map(bundle => {
            const firstRoute = remixConfig.routes[bundle.routes[0]];
            const config = resolvedConfigsMap.get(firstRoute) ?? {
                runtime: 'nodejs',
            };
            if (config.runtime === 'edge') {
                return createRenderEdgeFunction(entrypointFsDirname, repoRootPath, (0, path_1.join)(entrypointFsDirname, bundle.serverBuildPath), serverEntryPoint, remixVersion, config);
            }
            return createRenderNodeFunction(nodeVersion, entrypointFsDirname, repoRootPath, (0, path_1.join)(entrypointFsDirname, bundle.serverBuildPath), serverEntryPoint, remixVersion, config);
        }),
    ]);
    const output = staticFiles;
    const routes = [
        {
            src: '^/build/(.*)$',
            headers: { 'cache-control': 'public, max-age=31536000, immutable' },
            continue: true,
        },
        {
            handle: 'filesystem',
        },
    ];
    for (const route of remixRoutes) {
        // Layout routes don't get a function / route added
        if ((0, utils_1.isLayoutRoute)(route.id, remixRoutes))
            continue;
        const { path, rePath } = (0, utils_1.getPathFromRoute)(route, remixConfig.routes);
        // If the route is a pathless layout route (at the root level)
        // and doesn't have any sub-routes, then a function should not be created.
        if (!path) {
            continue;
        }
        const funcIndex = serverBundles.findIndex(bundle => {
            return bundle.routes.includes(route.id);
        });
        const func = functions[funcIndex];
        if (!func) {
            throw new Error(`Could not determine server bundle for "${route.id}"`);
        }
        output[path] =
            func instanceof build_utils_1.EdgeFunction
                ? // `EdgeFunction` currently requires the "name" property to be set.
                    // Ideally this property will be removed, at which point we can
                    // return the same `edgeFunction` instance instead of creating a
                    // new one for each page.
                    new build_utils_1.EdgeFunction({
                        ...func,
                        name: path,
                    })
                : func;
        // If this is a dynamic route then add a Vercel route
        const re = (0, utils_1.getRegExpFromPath)(rePath);
        if (re) {
            routes.push({
                src: re.source,
                dest: path,
            });
        }
    }
    // Add a 404 path for not found pages to be server-side rendered by Remix.
    // Use an edge function bundle if one was generated, otherwise use Node.js.
    if (!output['404']) {
        const edgeFunctionIndex = Array.from(serverBundlesMap.values()).findIndex(routes => {
            const runtime = resolvedConfigsMap.get(routes[0])?.runtime;
            return runtime === 'edge';
        });
        const func = edgeFunctionIndex !== -1 ? functions[edgeFunctionIndex] : functions[0];
        output['404'] =
            func instanceof build_utils_1.EdgeFunction
                ? new build_utils_1.EdgeFunction({ ...func, name: '404' })
                : func;
    }
    routes.push({
        src: '/(.*)',
        dest: '/404',
    });
    return { routes, output, framework: { version: remixVersion } };
};
exports.build = build;
function hasScript(scriptName, pkg) {
    const scripts = (pkg && pkg.scripts) || {};
    return typeof scripts[scriptName] === 'string';
}
async function createRenderNodeFunction(nodeVersion, entrypointDir, rootDir, serverBuildPath, serverEntryPoint, remixVersion, config) {
    const files = {};
    let handler = (0, path_1.relative)(rootDir, serverBuildPath);
    let handlerPath = (0, path_1.join)(rootDir, handler);
    if (!serverEntryPoint) {
        const baseServerBuildPath = (0, path_1.basename)(serverBuildPath, '.js');
        handler = (0, path_1.join)((0, path_1.dirname)(handler), `server-${baseServerBuildPath}.mjs`);
        handlerPath = (0, path_1.join)(rootDir, handler);
        // Copy the `server-node.mjs` file into the "build" directory
        const nodeServerSrc = await nodeServerSrcPromise;
        await writeEntrypointFile(handlerPath, nodeServerSrc.replace('@remix-run/dev/server-build', `./${baseServerBuildPath}.js`), rootDir);
    }
    // Trace the handler with `@vercel/nft`
    const trace = await (0, nft_1.nodeFileTrace)([handlerPath], {
        base: rootDir,
        processCwd: entrypointDir,
    });
    for (const warning of trace.warnings) {
        (0, build_utils_1.debug)(`Warning from trace: ${warning.message}`);
    }
    for (const file of trace.fileList) {
        files[file] = await build_utils_1.FileFsRef.fromFsPath({ fsPath: (0, path_1.join)(rootDir, file) });
    }
    const fn = new build_utils_1.NodejsLambda({
        files,
        handler,
        runtime: nodeVersion.runtime,
        shouldAddHelpers: false,
        shouldAddSourcemapSupport: false,
        operationType: 'SSR',
        supportsResponseStreaming: true,
        regions: config.regions,
        memory: config.memory,
        maxDuration: config.maxDuration,
        framework: {
            slug: 'remix',
            version: remixVersion,
        },
    });
    return fn;
}
async function createRenderEdgeFunction(entrypointDir, rootDir, serverBuildPath, serverEntryPoint, remixVersion, config) {
    const files = {};
    let handler = (0, path_1.relative)(rootDir, serverBuildPath);
    let handlerPath = (0, path_1.join)(rootDir, handler);
    if (!serverEntryPoint) {
        const baseServerBuildPath = (0, path_1.basename)(serverBuildPath, '.js');
        handler = (0, path_1.join)((0, path_1.dirname)(handler), `server-${baseServerBuildPath}.mjs`);
        handlerPath = (0, path_1.join)(rootDir, handler);
        // Copy the `server-edge.mjs` file into the "build" directory
        const edgeServerSrc = await edgeServerSrcPromise;
        await writeEntrypointFile(handlerPath, edgeServerSrc.replace('@remix-run/dev/server-build', `./${baseServerBuildPath}.js`), rootDir);
    }
    let remixRunVercelPkgJson;
    // Trace the handler with `@vercel/nft`
    const trace = await (0, nft_1.nodeFileTrace)([handlerPath], {
        base: rootDir,
        processCwd: entrypointDir,
        conditions: ['edge-light', 'browser', 'module', 'import', 'require'],
        async readFile(fsPath) {
            let source;
            try {
                source = await fs_1.promises.readFile(fsPath);
            }
            catch (err) {
                if (err.code === 'ENOENT' || err.code === 'EISDIR') {
                    return null;
                }
                throw err;
            }
            if ((0, path_1.basename)(fsPath) === 'package.json') {
                // For Edge Functions, patch "main" field to prefer "browser" or "module"
                const pkgJson = JSON.parse(source.toString());
                // When `@remix-run/vercel` is detected, we need to modify the `package.json`
                // to include the "browser" field so that the proper Edge entrypoint file
                // is used. This is a temporary stop gap until this PR is merged:
                // https://github.com/remix-run/remix/pull/5537
                if (pkgJson.name === '@remix-run/vercel') {
                    pkgJson.browser = 'dist/edge.js';
                    pkgJson.dependencies['@remix-run/server-runtime'] =
                        pkgJson.dependencies['@remix-run/node'];
                    if (!remixRunVercelPkgJson) {
                        remixRunVercelPkgJson = JSON.stringify(pkgJson, null, 2) + '\n';
                        // Copy in the edge entrypoint so that NFT can properly resolve it
                        const vercelEdgeEntrypointPath = (0, path_1.join)(DEFAULTS_PATH, 'vercel-edge-entrypoint.js');
                        const vercelEdgeEntrypointDest = (0, path_1.join)((0, path_1.dirname)(fsPath), 'dist/edge.js');
                        await fs_1.promises.copyFile(vercelEdgeEntrypointPath, vercelEdgeEntrypointDest);
                    }
                }
                for (const prop of ['browser', 'module']) {
                    const val = pkgJson[prop];
                    if (typeof val === 'string') {
                        pkgJson.main = val;
                        // Return the modified `package.json` to nft
                        source = JSON.stringify(pkgJson);
                        break;
                    }
                }
            }
            return source;
        },
    });
    for (const warning of trace.warnings) {
        (0, build_utils_1.debug)(`Warning from trace: ${warning.message}`);
    }
    for (const file of trace.fileList) {
        if (remixRunVercelPkgJson &&
            file.endsWith(`@remix-run${path_1.sep}vercel${path_1.sep}package.json`)) {
            // Use the modified `@remix-run/vercel` package.json which contains "browser" field
            files[file] = new build_utils_1.FileBlob({ data: remixRunVercelPkgJson });
        }
        else {
            files[file] = await build_utils_1.FileFsRef.fromFsPath({ fsPath: (0, path_1.join)(rootDir, file) });
        }
    }
    const fn = new build_utils_1.EdgeFunction({
        files,
        deploymentTarget: 'v8-worker',
        name: 'render',
        entrypoint: handler,
        regions: config.regions,
        framework: {
            slug: 'remix',
            version: remixVersion,
        },
    });
    return fn;
}
async function writeEntrypointFile(path, data, rootDir) {
    try {
        await fs_1.promises.writeFile(path, data);
    }
    catch (err) {
        if (err.code === 'ENOENT') {
            throw new Error(`The "${(0, path_1.relative)(rootDir, (0, path_1.dirname)(path))}" directory does not exist. Please contact support at https://vercel.com/help.`);
        }
        throw err;
    }
}
//# sourceMappingURL=build.js.map