"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const lodash_camelcase_1 = __importDefault(require("lodash.camelcase"));
const types_1 = require("./types");
function getFlagsProxy(ldClient, rawFlags, reactOptions = types_1.defaultReactOptions, targetFlags) {
    const filteredFlags = filterFlags(rawFlags, targetFlags);
    const [flags, flagKeyMap = {}] = reactOptions.useCamelCaseFlagKeys
        ? getCamelizedKeysAndFlagMap(filteredFlags)
        : [filteredFlags];
    return {
        flags: reactOptions.sendEventsOnFlagRead ? toFlagsProxy(ldClient, flags, flagKeyMap) : flags,
        flagKeyMap,
    };
}
exports.default = getFlagsProxy;
function filterFlags(flags, targetFlags) {
    if (targetFlags === undefined) {
        return flags;
    }
    return Object.keys(targetFlags).reduce((acc, key) => {
        if (hasFlag(flags, key)) {
            acc[key] = flags[key];
        }
        return acc;
    }, {});
}
function getCamelizedKeysAndFlagMap(rawFlags) {
    const flags = {};
    const flagKeyMap = {};
    for (const rawFlag in rawFlags) {
        // Exclude system keys
        if (rawFlag.indexOf('$') === 0) {
            continue;
        }
        const camelKey = (0, lodash_camelcase_1.default)(rawFlag);
        flags[camelKey] = rawFlags[rawFlag];
        flagKeyMap[camelKey] = rawFlag;
    }
    return [flags, flagKeyMap];
}
function hasFlag(flags, flagKey) {
    return Object.prototype.hasOwnProperty.call(flags, flagKey);
}
function toFlagsProxy(ldClient, flags, flagKeyMap) {
    return new Proxy(flags, {
        // trap for reading a flag value using `LDClient#variation` to trigger an evaluation event
        get(target, prop, receiver) {
            const currentValue = Reflect.get(target, prop, receiver);
            // only process flag keys and ignore symbols and native Object functions
            if (typeof prop === 'symbol' || !hasFlag(flagKeyMap, prop)) {
                return currentValue;
            }
            if (currentValue === undefined) {
                return;
            }
            return ldClient.variation(flagKeyMap[prop], currentValue);
        },
        // disable all mutation functions to make proxy readonly
        setPrototypeOf: () => false,
        set: () => false,
        defineProperty: () => false,
        deleteProperty: () => false,
        preventExtensions: () => false,
    });
}
//# sourceMappingURL=getFlagsProxy.js.map