rrule.js
========

**Library for working with recurrence rules for calendar dates.**

[![NPM version][npm-image]][npm-url]
[![Build Status][travis-image]][travis-url]
[![js-standard-style][js-standard-image]][js-standard-url]
[![Downloads][downloads-image]][downloads-url]
[![Gitter][gitter-image]][gitter-url]
[![codecov.io](http://codecov.io/github/jakubroztocil/rrule/coverage.svg?branch=master)](http://codecov.io/github/jakubroztocil/rrule?branch=master)

rrule.js supports recurrence rules as defined in the [iCalendar
RFC](http://www.ietf.org/rfc/rfc2445.txt), with a few important
[differences](#differences-from-icalendar-rfc). It is a partial port of the
`rrule` module from the excellent
[python-dateutil](http://labix.org/python-dateutil/) library. On top of
that, it supports parsing and serialization of recurrence rules from and
to natural language.

* * * * *


### Quick Start

-   [Demo app](http://jakubroztocil.github.io/rrule/)

#### Client Side

```bash
$ yarn add rrule
```

Alternatively, download manually:

 * [rrule.min.js](https://raw.github.com/jakubroztocil/rrule/master/dist/es5/rrule.min.js) (bundled, minified)
 * [rrule.js](https://raw.github.com/jakubroztocil/rrule/master/dist/es5/rrule.js) (bundled, not minified)
 * [rrule-tz.min.js](https://raw.github.com/jakubroztocil/rrule/master/dist/es5/rrule-tz.min.js) (with timezone support, bundled, minified)
 * [rrule-tz.js](https://raw.github.com/jakubroztocil/rrule/master/dist/es5/rrule-tz.js) (with timezone support, bundled, not minified)
 * [rrule.js](https://raw.github.com/jakubroztocil/rrule/master/dist/esm/rrule.js) (ES module source with optional TypeScript types)

```html
<script src="rrule/dist/es5/rrule.min.js"></script>
```

#### Server Side

```bash
$ yarn add rrule
```

```es6
import { RRule, RRuleSet, rrulestr } from 'rrule'
```

#### Usage

**RRule:**
```js
// Create a rule:
const rule = new RRule({
  freq: RRule.WEEKLY,
  interval: 5,
  byweekday: [RRule.MO, RRule.FR],
  dtstart: new Date(Date.UTC(2012, 1, 1, 10, 30)),
  until: new Date(Date.UTC(2012, 12, 31))
})

// Get all occurrence dates (Date instances):
rule.all()
[ '2012-02-03T10:30:00.000Z',
  '2012-03-05T10:30:00.000Z',
  '2012-03-09T10:30:00.000Z',
  '2012-04-09T10:30:00.000Z',
  '2012-04-13T10:30:00.000Z',
  '2012-05-14T10:30:00.000Z',
  '2012-05-18T10:30:00.000Z',

 /* … */]

// Get a slice:
rule.between(new Date(Date.UTC(2012, 7, 1)), new Date(Date.UTC(2012, 8, 1)))
['2012-08-27T10:30:00.000Z',
 '2012-08-31T10:30:00.000Z']

// Get an iCalendar RRULE string representation:
// The output can be used with RRule.fromString().
rule.toString()
"FREQ=WEEKLY;DTSTART=20120201T093000Z;INTERVAL=5;UNTIL=20130130T230000Z;BYDAY=MO,FR"

// Get a human-friendly text representation:
// The output can be used with RRule.fromText().
rule.toText()
"every 5 weeks on Monday, Friday until January 31, 2013"
```

**RRuleSet:**
```js
const rruleSet = new RRuleSet()

// Add a rrule to rruleSet
rruleSet.rrule(new RRule({
  freq: RRule.MONTHLY,
  count: 5,
  dtstart: new Date(Date.UTC(2012, 1, 1, 10, 30))
}))

// Add a date to rruleSet
rruleSet.rdate(new Date(Date.UTC(2012, 6, 1, 10, 30)))

// Add another date to rruleSet
rruleSet.rdate(new Date(Date.UTC(2012, 6, 2, 10, 30)))

// Add a exclusion rrule to rruleSet
rruleSet.exrule(new r.RRule({
  freq: RRule.MONTHLY,
  count: 2,
  dtstart: new Date(Date.UTC(2012, 2, 1, 10, 30))
}))

// Add a exclusion date to rruleSet
rruleSet.exdate(new Date(Date.UTC(2012, 5, 1, 10, 30)))

// Get all occurrence dates (Date instances):
rruleSet.all()
[ '2012-02-01T10:30:00.000Z',
  '2012-05-01T10:30:00.000Z',
  '2012-07-01T10:30:00.000Z',
  '2012-07-02T10:30:00.000Z' ]

// Get a slice:
rruleSet.between(new Date(Date.UTC(2012, 2, 1)), new Date(Date.UTC(2012, 6, 2)))
[ '2012-05-01T10:30:00.000Z', '2012-07-01T10:30:00.000Z' ]


 // To string
rruleSet.valueOf()
['RRULE:FREQ=MONTHLY;COUNT=5;DTSTART=20120201T023000Z',
 'RDATE:20120701T023000Z,20120702T023000Z',
 'EXRULE:FREQ=MONTHLY;COUNT=2;DTSTART=20120301T023000Z',
 'EXDATE:20120601T023000Z']

// To string
rruleSet.toString()
'["RRULE:FREQ=MONTHLY;COUNT=5;DTSTART=20120201T023000Z","RDATE:20120701T023000Z,20120702T023000Z","EXRULE:FREQ=MONTHLY;COUNT=2;DTSTART=20120301T023000Z","EXDATE:20120601T023000Z"]'
```

**rrulestr:**
```js
// Parse a RRule string, return a RRule object
rrulestr('RRULE:FREQ=MONTHLY;COUNT=5;DTSTART=20120201T023000Z')

// Parse a RRule string, return a RRuleSet object
rrulestr('RRULE:FREQ=MONTHLY;COUNT=5;DTSTART=20120201T023000Z', {forceset: true})

// Parse a RRuleSet string, return a RRuleSet object
rrulestr('RRULE:FREQ=MONTHLY;COUNT=5;DTSTART=20120201T023000Z\nRDATE:20120701T023000Z,20120702T023000Z\nEXRULE:FREQ=MONTHLY;COUNT=2;DTSTART=20120301T023000Z\nEXDATE:20120601T023000Z')

```

For more examples see
[python-dateutil](http://labix.org/python-dateutil/) documentation.

* * * * *

### Timezone Support

By default, `RRule` only correctly supports
["floating" times or UTC timezones](https://tools.ietf.org/html/rfc2445#section-4.2.19).
Optionally, it also supports use of the `TZID` parameter in the
[RFC](https://tools.ietf.org/html/rfc2445#section-4.2.19)
when the [Luxon](https://github.com/moment/luxon) library is provided. The 
[specification](https://moment.github.io/luxon/docs/manual/zones.html#specifying-a-zone)
and [support matrix](https://moment.github.io/luxon/docs/manual/matrix.html) for Luxon apply.

Example with `TZID`:

```js
new RRule({
  dtstart: new Date(Date.UTC(2018, 1, 1, 10, 30)),
  count: 1,
  tzid: 'Asia/Tokyo'
}).all()

// assuming the system timezone is set to America/Los_Angeles, you get:
[ '2018-01-31T17:30:00.000Z' ]
// which is the time in Los Angeles when it's 2018-02-01T10:30:00 in Tokyo.
```

Whether or not you use the `TZID` param, make sure to only use JS `Date` objects that are
represented in UTC to avoid unexpected timezone offsets being applied, for example:

```js
// WRONG: Will produce dates with TZ offsets added
new RRule({
  freq: RRule.MONTHLY,
  dtstart: new Date(2018, 1, 1, 10, 30),
  until: new Date(2018, 2, 31)
}).all()

[ '2018-02-01T18:30:00.000Z' ]

// RIGHT: Will produce dates with recurrences at the correct time
new RRule({
  freq: RRule.MONTHLY,
  dtstart: new Date(Date.UTC(2018, 1, 1, 10, 30)),
  until: new Date(Date.UTC(2018, 2, 31))
}).all()

[ '2018-02-01T10:30:00.000Z' ]
```

### API

#### `RRule` Constructor

```javascript
new RRule(options[, noCache=false])
```

The `options` argument mostly corresponds to the properties defined for `RRULE` in the
iCalendar RFC. Only `freq` is required.

<table>
    <!-- why, markdown... -->
    <thead>
    <tr>
        <th>Option</th>
        <th>Description</th>
    </tr>
    <thead>
    <tbody>
    <tr>
        <td><code>freq</code></td>
        <td>
            <p>(required) One of the following constants:</p>
            <ul>
                <li><code>RRule.YEARLY</code></li>
                <li><code>RRule.MONTHLY</code></li>
                <li><code>RRule.WEEKLY</code></li>
                <li><code>RRule.DAILY</code></li>
                <li><code>RRule.HOURLY</code></li>
                <li><code>RRule.MINUTELY</code></li>
                <li><code>RRule.SECONDLY</code></li>
            </ul>
        </td>
    </tr>
    <tr>
        <td><code>dtstart</code></td>
        <td>The recurrence start. Besides being the base for the
            recurrence, missing parameters in the final recurrence
            instances will also be extracted from this date. If not
            given, <code>new Date</code> will be used instead.
        </td>
    </tr>
    <tr>
        <td><code>interval</code></td>
        <td>The interval between each freq iteration. For example,
            when using <code>RRule.YEARLY</code>, an interval of <code>2</code> means
            once every
            two years, but with <code>RRule.HOURLY</code>, it means once every two
            hours.
            The default interval is <code>1</code>.
        </td>
    </tr>
    <tr>
        <td><code>wkst</code></td>
        <td>The week start day. Must be one of the <code>RRule.MO</code>,
            <code>RRule.TU</code>, <code>RRule.WE</code> constants, or an integer,
            specifying
            the first day of the week. This will affect recurrences based
            on weekly periods. The default week start is <code>RRule.MO</code>.
        </td>
    </tr>
    <tr>
        <td><code>count</code></td>
        <td>How many occurrences will be generated.</td>
    </tr>
    <tr>
        <td><code>until</code></td>
        <td>If given, this must be a <code>Date</code> instance, that will specify
            the limit of the recurrence. If a recurrence instance happens
            to be the same as the <code>Date</code> instance given in the
            <code>until</code>
            argument, this will be the last occurrence.
        </td>
    </tr>
    <tr>
      <td><code>tzid</code></td>
      <td>If given, this must be a string <a href="https://moment.github.io/luxon/docs/manual/zones.html#specifying-a-zone">supported</a>
      by Luxon, and the <a href="https://moment.github.io/luxon/">Luxon</a> library must be provided. See
      discussion under <a href="#timezone-support">Timezone support</a>.
      </td>
    </tr>
    <tr>
        <td><code>bysetpos</code></td>
        <td>If given, it must be either an integer, or an array of
            integers, positive or negative. Each given integer will specify
            an occurrence number, corresponding to the nth occurrence of
            the rule inside the frequency period. For example, a
            <code>bysetpos</code> of <code>-1</code> if combined with a <code>RRule.MONTHLY</code>
            frequency, and a byweekday of (<code>RRule.MO</code>, <code>RRule.TU</code>,
            <code>RRule.WE</code>, <code>RRule.TH</code>, <code>RRule.FR</code>), will result in
            the last
            work day of every month.
        </td>
    </tr>
    <tr>
        <td><code>bymonth</code></td>
        <td>If given, it must be either an integer, or an array of
            integers, meaning the months to apply the recurrence to.
        </td>
    </tr>
    <tr>
        <td><code>bymonthday</code></td>
        <td>If given, it must be either an integer, or an array of
            integers, meaning the month days to apply the recurrence to.
        </td>
    </tr>
    <tr>
        <td><code>byyearday</code></td>
        <td>If given, it must be either an integer, or an array of
            integers, meaning the year days to apply the recurrence to.
        </td>
    </tr>
    <tr>
        <td><code>byweekno</code></td>
        <td>If given, it must be either an integer, or an array of
            integers, meaning the week numbers to apply the recurrence to.
            Week numbers have the meaning described in ISO8601, that is,
            the first week of the year is that containing at least four
            days of the new year.
        </td>
    </tr>
    <tr>
        <td><code>byweekday</code></td>
        <td>If given, it must be either an integer (<code>0 == RRule.MO</code>), an
            array of integers, one of the weekday constants
            (<code>RRule.MO</code>,
            <code>RRule.TU</code>, etc), or an array of these constants. When
            given,
            these variables will define the weekdays where the recurrence
            will be applied. It's also possible to use an argument n for
            the weekday instances, which will mean the nth occurrence of
            this weekday in the period. For example, with
            <code>RRule.MONTHLY</code>,
            or with <code>RRule.YEARLY</code> and <code>BYMONTH</code>, using
            <code>RRule.FR.nth(+1)</code> or <code>RRule.FR.nth(-1)</code> in <code>byweekday</code>
            will specify the first or last friday of the month where the
            recurrence happens.
            Notice
            that the RFC documentation, this is specified as <code>BYDAY</code>,
            but was renamed to avoid the ambiguity of that argument.
        </td>
    </tr>
    <tr>
        <td><code>byhour</code></td>
        <td>If given, it must be either an integer, or an array of
            integers, meaning the hours to apply the recurrence to.
        </td>
    </tr>
    <tr>
        <td><code>byminute</code></td>
        <td>If given, it must be either an integer, or an array of
            integers, meaning the minutes to apply the recurrence to.
        </td>
    </tr>
    <tr>
        <td><code>bysecond</code></td>
        <td>If given, it must be either an integer, or an array of
            integers, meaning the seconds to apply the recurrence to.
        </td>
    </tr>
    <tr>
        <td><code>byeaster</code></td>
        <td>This is an extension to the RFC specification which the Python
            implementation provides.
            <strong>Not implemented in the JavaScript version.</strong>
        </td>
    </tr>
    </tbody>
</table>


`noCache`: Set to `true` to disable caching of results. If you will use the
same rrule instance multiple times, enabling caching will improve the
performance considerably. Enabled by default.

See also [python-dateutil](http://labix.org/python-dateutil/)
documentation.


* * * * *


#### Instance properties

<dl>
    <dt><code>rule.options</code></dt>
    <dd>Processed options applied to the rule. Includes default options
    (such us <code>wkstart</code>). Currently,
    <code>rule.options.byweekday</code> isn't equal
    to <code>rule.origOptions.byweekday</code> (which is an inconsistency).
    </dd>
    <dt><code>rule.origOptions</code></dt>
    <dd>The original <code>options</code> argument passed to
    the constructor.</dd>
</dl>


* * * * *


#### Occurrence Retrieval Methods

##### `RRule.prototype.all([iterator])`

Returns all dates matching the rule. It is a replacement for the
iterator protocol this class implements in the Python version.

As rules without `until` or `count` represent infinite date series, you
can optionally pass `iterator`, which is a function that is called for
each date matched by the rule. It gets two parameters `date` (the `Date`
instance being added), and `i` (zero-indexed position of `date` in the
result). Dates are being added to the result as long as the iterator
returns `true`. If a `false`-y value is returned, `date` isn't added to
the result and the iteration is interrupted (possibly prematurely).

```javascript
rule.all()
[ '2012-02-01T10:30:00.000Z',
  '2012-05-01T10:30:00.000Z',
  '2012-07-01T10:30:00.000Z',
  '2012-07-02T10:30:00.000Z' ]

rule.all(function (date, i){return i < 2})
[ '2012-02-01T10:30:00.000Z',
  '2012-05-01T10:30:00.000Z' ]
```

##### `RRule.prototype.between(after, before, inc=false [, iterator])`

Returns all the occurrences of the rrule between `after` and `before`.
The inc keyword defines what happens if `after` and/or `before` are
themselves occurrences. With `inc == true`, they will be included in the
list, if they are found in the recurrence set.

Optional `iterator` has the same function as it has with
`RRule.prototype.all()`.

```javascript
rule.between(new Date(Date.UTC(2012, 7, 1)), new Date(Date.UTC(2012, 8, 1)))
['2012-08-27T10:30:00.000Z',
 '2012-08-31T10:30:00.000Z']
```

##### `RRule.prototype.before(dt, inc=false)`

Returns the last recurrence before the given `Date` instance. The `inc`
argument defines what happens if `dt` is an occurrence. With
`inc == true`, if `dt` itself is an occurrence, it will be returned.


##### `RRule.prototype.after(dt, inc=false)`

Returns the first recurrence
after the given `Date` instance. The `inc` argument defines what happens
if `dt` is an occurrence. With `inc == true`, if `dt` itself is an
occurrence, it will be returned.

See also [python-dateutil](http://labix.org/python-dateutil/)
documentation.


* * * * *


#### iCalendar RFC String Methods

##### `RRule.prototype.toString()`

Returns a string representation of the rule as per the iCalendar RFC.
Only properties explicitly specified in `options` are included:

```javascript
rule.toString()
"FREQ=WEEKLY;DTSTART=20120201T093000Z;INTERVAL=5;UNTIL=20130130T230000Z;BYDAY=MO,FR"

rule.toString() == RRule.optionsToString(rule.origOptions)
true
```

##### `RRule.optionsToString(options)`

Converts `options` to iCalendar RFC `RRULE` string:

```javascript
// Get full a string representation of all options,
// including the default and inferred ones.
RRule.optionsToString(rule.options)
"FREQ=WEEKLY;DTSTART=20120201T093000Z;INTERVAL=5;WKST=0;UNTIL=20130130T230000Z;BYDAY=MO,FR;BYHOUR=10;BYMINUTE=30;BYSECOND=0"

// Cherry-pick only some options from an rrule:
RRule.optionsToString({
  freq: rule.options.freq,
  dtstart: rule.options.dtstart
})
"FREQ=WEEKLY;DTSTART=20120201T093000Z"
```

##### `RRule.fromString(rfcString)`

Constructs an `RRule` instance from a complete `rfcString`:

```javascript
var rule = RRule.fromString("FREQ=WEEKLY;DTSTART=20120201T093000Z")

// This is equivalent
var rule = new RRule(RRule.parseString("FREQ=WEEKLY;DTSTART=20120201T093000Z"))
```

##### `RRule.parseString(rfcString)`

Only parse RFC string and return `options`.

```javascript
var options = RRule.parseString('FREQ=DAILY;INTERVAL=6')
options.dtstart = new Date(Date.UTC(2000, 1, 1))
var rule = new RRule(options)
```

* * * * *

#### Natural Language Text Methods

These methods provide an incomplete support for text–`RRule` and
`RRule`–text conversion. You should test them with your input to see
whether the result is acceptable.

##### `RRule.prototype.toText([gettext, [language]])`

Returns a textual representation of `rule`. The `gettext` callback, if
provided, will be called for each text token and its return value used
instead. The optional `language` argument is a language definition to be
used (defaults to `rrule/nlp.js:ENGLISH`).

```javascript
var rule = new RRule({
  freq: RRule.WEEKLY,
  count: 23
})
rule.toText()
"every week for 23 times"
```

##### `RRule.prototype.isFullyConvertibleToText()`

Provides a hint on whether all the options the rule has are convertible
to text.

##### `RRule.fromText(text[, language])`

Constructs an `RRule` instance from `text`.

```javascript
rule = RRule.fromText('every day for 3 times')
```

##### `RRule.parseText(text[, language])`

Parse `text` into `options`:

```javascript
options = RRule.parseText('every day for 3 times')
// {freq: 3, count: "3"}
options.dtstart = new Date(2000, 1, 1)
var rule = new RRule(options)
```


* * * * *

#### `RRuleSet` Constructor

```javascript
new RRuleSet([noCache=false])
```

The RRuleSet instance allows more complex recurrence setups, mixing multiple
 rules, dates, exclusion rules, and exclusion dates.

Default `noCache` argument is `false`, caching of results will be enabled,
improving performance of multiple queries considerably.

##### `RRuleSet.prototype.rrule(rrule)`

Include the given rrule instance in the recurrence set generation.

##### `RRuleSet.prototype.rdate(dt)`
Include the given datetime instance in the recurrence set generation.

##### `RRuleSet.prototype.exrule(rrule)`
Include the given rrule instance in the recurrence set exclusion list. Dates
which are part of the given recurrence rules will not be generated, even if
some inclusive rrule or rdate matches them.

##### `RRuleSet.prototype.exdate(dt)`
Include the given datetime instance in the recurrence set exclusion list. Dates
included that way will not be generated, even if some inclusive rrule or
rdate matches them.

##### `RRuleSet.prototype.all([iterator])`

Same as `RRule.prototype.all`.

##### `RRuleSet.prototype.between(after, before, inc=false [, iterator])`

Same as `RRule.prototype.between`.

##### `RRuleSet.prototype.before(dt, inc=false)`

Same as `RRule.prototype.before`.

##### `RRuleSet.prototype.after(dt, inc=false)`

Same as `RRule.prototype.after`.

* * * * *

#### `rrulestr` Function

```js
rrulestr(rruleStr[, options])
```

The `rrulestr` function is a parser for RFC-like syntaxes. The string passed
as parameter may be a multiple line string, a single line string, or just the
RRULE property value.

Additionally, it accepts the following keyword arguments:

`cache`
If True, the rruleset or rrule created instance will cache its results.
Default is not to cache.

`dtstart`
If given, it must be a datetime instance that will be used when no DTSTART
property is found in the parsed string. If it is not given, and the property
is not found, datetime.now() will be used instead.

`unfold`
If set to True, lines will be unfolded following the RFC specification. It
defaults to False, meaning that spaces before every line will be stripped.

`forceset`
If set to True a rruleset instance will be returned, even if only a single rule
is found. The default is to return an rrule if possible, and an rruleset if necessary.

`compatible`
If set to True, the parser will operate in RFC-compatible mode. Right now it
means that unfold will be turned on, and if a DTSTART is found, it will be
considered the first recurrence instance, as documented in the RFC.

`tzid`
If given, it must be a string that will be used when no `TZID` property is found
in the parsed string. If it is not given, and the property is not found, `'UTC'`
will be used by default.


* * * * *

### Differences From iCalendar RFC

* `RRule` has no `byday` keyword. The equivalent keyword has been replaced by
the `byweekday` keyword, to remove the ambiguity present in the original
keyword.
* Unlike documented in the RFC, the starting datetime, `dtstart`, is
not the first recurrence instance, unless it does fit in the specified rules.
This is in part due to this project being a port of
[python-dateutil](https://labix.org/python-dateutil#head-a65103993a21b717f6702063f3717e6e75b4ba66),
which has the same non-compliant functionality. Note that you can get the
original behavior by using a `RRuleSet` and adding the `dtstart` as an `rdate`.

```javascript
var rruleSet = new RRuleSet()
var start = new Date(Date.UTC(2012, 1, 1, 10, 30))

// Add a rrule to rruleSet
rruleSet.rrule(new RRule({
  freq: RRule.MONTHLY,
  count: 5,
  dtstart: start
}))

// Add a date to rruleSet
rruleSet.rdate(start)
```

* Unlike documented in the RFC, every keyword is valid on every frequency (the
RFC documents that `byweekno` is only valid on yearly frequencies, for example).

### Development

rrule.js is implemented in Typescript. It uses [JavaScript Standard Style](https://github.com/feross/standard) coding style.

To run the code, checkout this repository and run:

```
$ yarn
```

To run the tests, run:

```
$ yarn test
```

To build files for distribution, run:

```
$ yarn build
```

#### Authors

* [Jakub Roztocil](http://roztocil.co/)
    ([@jakubroztocil](http://twitter.com/jakubroztocil))
* Lars Schöning ([@lyschoening](http://twitter.com/lyschoening))

Python `dateutil` is written by [Gustavo
Niemeyer](http://niemeyer.net/).

See [LICENCE](https://github.com/jakubroztocil/rrule/blob/master/LICENCE) for
more details.

[npm-url]: https://npmjs.org/package/rrule
[npm-image]: http://img.shields.io/npm/v/rrule.svg

[travis-url]: https://travis-ci.org/jakubroztocil/rrule
[travis-image]: http://img.shields.io/travis/jakubroztocil/rrule.svg

[downloads-url]: https://npmjs.org/package/rrule
[downloads-image]: http://img.shields.io/npm/dm/rrule.svg?style=flat-square

[js-standard-url]: https://github.com/feross/standard
[js-standard-image]: https://img.shields.io/badge/code%20style-standard-brightgreen.svg?style=flat

[gitter-url]: https://gitter.im/rrule-js/Lobby
[gitter-image]: https://img.shields.io/gitter/room/nwjs/nw.js.svg
