"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.sameModule = exports.maybeRelativePath = exports.emitImports = exports.SideEffect = exports.Augmented = exports.ImportsAll = exports.ImportsDefault = exports.ImportsName = exports.Imported = exports.Implicit = exports.Import = exports.importType = void 0;
const path = __importStar(require("path"));
const Node_1 = require("./Node");
const utils_1 = require("./utils");
const typeImportMarker = "(?:t:)?";
const fileNamePattern = "(?:[a-zA-Z0-9._-]+)";
const modulePattern = `@?(?:(?:${fileNamePattern}(?:/${fileNamePattern})*))`;
const identPattern = `(?:(?:[a-zA-Z][_a-zA-Z0-9.]*)|(?:[_a-zA-Z][_a-zA-Z0-9.]+))`;
exports.importType = "[*@+=]";
const importPattern = `^(${typeImportMarker}${identPattern})?(${exports.importType})(${modulePattern})(?:#(${identPattern}))?`;
const sourceIdentPattern = `(?:(?:${identPattern}:)?)`;
const sourceImportPattern = `^(${typeImportMarker}${sourceIdentPattern}${identPattern})?(@)(${modulePattern})(?:#(${identPattern}))?`;
/**
 * Specifies a symbol and its related origin, either via import or implicit/local declaration.
 */
class Import extends Node_1.Node {
    constructor(symbol) {
        super();
        this.symbol = symbol;
    }
    /**
     * Parses a symbol reference pattern to create a symbol. The pattern
     * allows the simple definition of all symbol types including any possible
     * import variation. If the spec to parse does not follow the proper format
     * an implicit symbol is created from the unparsed spec.
     *
     * Pattern: `symbolName? importType modulePath (#<augmentedSymbolName>)?`
     *
     * Where:
     *
     * - `symbolName` is any legal JS/TS symbol. If none, we use the last part of the module path as a guess.
     * - `importType` is one of `@` or `*` or `+`, where:
     *    - `@` is a named import
     *       - `Foo@bar` becomes `import { Foo } from 'bar'`
     *    - `*` is a star import,
     *       - `*Foo` becomes `import * as Foo from 'Foo'`
     *       - `Foo*foo` becomes `import * as Foo from 'foo'`
     *    - `+` is an implicit import
     *       - E.g. `Foo+foo` becomes `import 'foo'`
     * - `modulePath` is a path
     *    - E.g. `<filename>(/<filename)*`
     * - augmentedSymbolName = `[a-zA-Z0-9_]+`
     *
     *        Any valid symbol name that represents the symbol that is being augmented. For example,
     *        the import `rxjs/add/observable/from` attaches the `from` method to the `Observable` class.
     *        To import it correctly the spec should be `+rxjs/add/observable/from#Observable`. Adding this
     *        parameter to augmented imports ensures they are output only when the symbol being augmented
     *        is actually used.
     *
     *
     * @param spec Symbol spec to parse.
     * @return Parsed symbol specification
     */
    static from(spec) {
        let matched = spec.match(importPattern);
        if (matched === null) {
            matched = spec.match(sourceImportPattern);
        }
        if (matched != null) {
            const modulePath = matched[3];
            const kind = matched[2] || "@";
            const symbolName = matched[1] || (0, utils_1.last)(modulePath.split("/")) || "";
            const targetName = matched[4];
            switch (kind) {
                case "*":
                    return Import.importsAll(symbolName, modulePath);
                case "@":
                    const isTypeImport = symbolName.startsWith("t:");
                    let exportedNames;
                    if (isTypeImport) {
                        exportedNames = symbolName.substring(2).split(":");
                    }
                    else {
                        exportedNames = symbolName.split(":");
                    }
                    const exportedName = exportedNames.pop();
                    const sourceExportedName = exportedNames[0];
                    return Import.importsName(exportedName, modulePath, isTypeImport, sourceExportedName);
                case "=":
                    return Import.importsDefault(symbolName, modulePath);
                case "+":
                    return targetName
                        ? Import.augmented(symbolName, modulePath, targetName)
                        : Import.sideEffect(symbolName, modulePath);
                default:
                    throw new Error("Invalid import kind character");
            }
        }
        return Import.implicit(spec);
    }
    static fromMaybeString(spec) {
        return typeof spec === "string" ? Import.from(spec) : spec;
    }
    toCodeString() {
        return this.symbol;
    }
    get childNodes() {
        return [];
    }
    /**
     * Creates an import of all the modules exported symbols as a single
     * local named symbol
     *
     * e.g. `import * as Engine from 'templates';`
     *
     * @param localName The local name of the imported symbols
     * @param from The module to import the symbols from
     */
    static importsAll(localName, from) {
        return new ImportsAll(localName, from);
    }
    /**
     * Creates an import of a single named symbol from the module's exported
     * symbols.
     *
     * e.g. `import { Engine } from 'templates';`
     *
     * @param exportedName The symbol that is both exported and imported
     * @param from The module the symbol is exported from
     * @param typeImport whether this is an `import type` import
     */
    static importsName(exportedName, from, typeImport, sourceExportedName) {
        return new ImportsName(exportedName, from, sourceExportedName, typeImport);
    }
    /**
     * Creates a symbol that is brought in by a whole module import
     * that "augments" an existing symbol.
     *
     * e.g. `import 'rxjs/add/operator/flatMap'`
     *
     * @param symbolName The augmented symbol to be imported
     * @param from The entire import that does the augmentation
     * @param target The symbol that is augmented
     */
    static augmented(symbolName, from, target) {
        return new Augmented(symbolName, from, target);
    }
    /**
     * Creates a symbol that is brought in as a side effect of
     * an import.
     *
     * e.g. `import 'mocha'`
     *
     * @param symbolName The symbol to be imported
     * @param from The entire import that does the augmentation
     */
    static sideEffect(symbolName, from) {
        return new SideEffect(symbolName, from);
    }
    /**
     * An implied symbol that does no tracking of imports
     *
     * @param name The implicit symbol name
     */
    static implicit(name) {
        return new Implicit(name);
    }
    /**
     * Creates an import of a single named symbol from the module's exported
     * default.
     *
     * e.g. `import Engine from 'engine';`
     *
     * @param exportedName The symbol that is both exported and imported
     * @param from The module the symbol is exported from
     */
    static importsDefault(exportedName, from) {
        return new ImportsDefault(exportedName, from);
    }
}
exports.Import = Import;
/**
 * Non-imported symbol
 */
class Implicit extends Import {
    constructor(symbol) {
        super(symbol);
        this.source = undefined;
    }
}
exports.Implicit = Implicit;
/** Common base class for imported symbols. */
class Imported extends Import {
    /** The symbol is the imported symbol, i.e. `BarClass`, and source is the path it comes from. */
    constructor(symbol, source) {
        super(source);
        this.symbol = symbol;
        this.source = source;
    }
}
exports.Imported = Imported;
/**
 * Imports a single named symbol from the module's exported
 * symbols.
 *
 * E.g.:
 *
 * `import { Engine } from 'templates'` or
 * `import { Engine as Engine2 } from 'templates'`
 */
class ImportsName extends Imported {
    /**
     * @param symbol
     * @param source
     * @param sourceSymbol is the optional original symbol, i.e if we're renaming the symbol it is `Engine`
     * @param typeImport whether this is an `import type` import
     */
    constructor(symbol, source, sourceSymbol, typeImport) {
        super(symbol, source);
        this.sourceSymbol = sourceSymbol;
        this.typeImport = typeImport;
    }
    toImportPiece() {
        return this.sourceSymbol ? `${this.sourceSymbol} as ${this.symbol}` : this.symbol;
    }
}
exports.ImportsName = ImportsName;
/**
 * Imports a single named symbol from the module's exported
 * default.
 *
 * e.g. `import Engine from 'engine';`
 */
class ImportsDefault extends Imported {
    constructor(symbol, source) {
        super(symbol, source);
    }
}
exports.ImportsDefault = ImportsDefault;
/**
 * Imports all of the modules exported symbols as a single
 * named symbol
 *
 * e.g. `import * as Engine from 'templates';`
 */
class ImportsAll extends Imported {
    constructor(symbol, source) {
        super(symbol, source);
    }
}
exports.ImportsAll = ImportsAll;
/**
 * A symbol that is brought in by a whole module import
 * that "augments" an existing symbol.
 *
 * e.g. `import 'rxjs/add/operator/flatMap'`
 */
class Augmented extends Imported {
    constructor(symbol, source, augmented) {
        super(symbol, source);
        this.augmented = augmented;
    }
}
exports.Augmented = Augmented;
/**
 * A symbol that is brought in as a side effect of an import.
 *
 * E.g. `from("Foo+mocha")` will add `import 'mocha'`
 */
class SideEffect extends Imported {
    constructor(symbol, source) {
        super(symbol, source);
    }
}
exports.SideEffect = SideEffect;
/** Generates the `import ...` lines for the given `imports`. */
function emitImports(imports, ourModulePath, importMappings) {
    if (imports.length == 0) {
        return "";
    }
    let result = "";
    const augmentImports = (0, utils_1.groupBy)(filterInstances(imports, Augmented), (a) => a.augmented);
    // Group the imports by source module they're imported from
    const importsByModule = (0, utils_1.groupBy)(imports.filter((it) => it.source !== undefined &&
        // Ignore imports that are in our own file
        !(it instanceof ImportsName && it.definedIn && sameModule(it.definedIn, ourModulePath))), (it) => it.source);
    // Output each source module as one line
    Object.entries(importsByModule).forEach(([modulePath, imports]) => {
        // Skip imports from the current module
        if (sameModule(ourModulePath, modulePath)) {
            return;
        }
        if (modulePath in importMappings) {
            modulePath = importMappings[modulePath];
        }
        const importPath = maybeRelativePath(ourModulePath, modulePath);
        // Output star imports individually
        unique(filterInstances(imports, ImportsAll).map((i) => i.symbol)).forEach((symbol) => {
            result += `import * as ${symbol} from '${importPath}';\n`;
            const augments = augmentImports[symbol];
            if (augments) {
                augments.forEach((augment) => (result += `import '${augment.source}';\n`));
            }
        });
        // Partition named imported into `import type` vs. regular imports
        const allNames = filterInstances(imports, ImportsName);
        const names = unique(allNames.filter((i) => !i.typeImport).map((it) => it.toImportPiece()));
        const def = unique(filterInstances(imports, ImportsDefault).map((it) => it.symbol));
        // Output named imports as a group
        if (names.length > 0 || def.length > 0) {
            const namesPart = names.length > 0 ? [`{ ${names.join(", ")} }`] : [];
            const defPart = def.length > 0 ? [def[0]] : [];
            result += `import ${[...defPart, ...namesPart].join(", ")} from '${importPath}';\n`;
            [...names, ...def].forEach((name) => {
                const augments = augmentImports[name];
                if (augments) {
                    augments.forEach((augment) => (result += `import '${augment.source}';\n`));
                }
            });
        }
        const typeImports = unique(allNames
            .filter((i) => i.typeImport)
            .map((it) => it.toImportPiece())
            // If the `import type` is already used as a concrete import, just use that
            .filter((p) => !names.includes(p)));
        if (typeImports.length > 0) {
            result += `import type { ${typeImports.join(", ")} } from '${importPath}';\n`;
        }
    });
    const sideEffectImports = (0, utils_1.groupBy)(filterInstances(imports, SideEffect), (a) => a.source);
    Object.keys(sideEffectImports).forEach((it) => (result += `import '${it}';\n`));
    return result;
}
exports.emitImports = emitImports;
function filterInstances(list, t) {
    return list.filter((e) => e instanceof t);
}
function unique(list) {
    return [...new Set(list)];
}
function maybeRelativePath(outputPath, importPath) {
    if (!importPath.startsWith("./")) {
        return importPath;
    }
    importPath = path.normalize(importPath);
    outputPath = path.normalize(outputPath);
    const outputPathDir = path.dirname(outputPath);
    let relativePath = path.relative(outputPathDir, importPath).split(path.sep).join(path.posix.sep);
    if (!relativePath.startsWith(".")) {
        // ensure the js compiler recognizes this is a relative path.
        relativePath = "./" + relativePath;
    }
    return relativePath;
}
exports.maybeRelativePath = maybeRelativePath;
/** Checks if `path1 === path2` despite minor path differences like `./foo` and `foo`. */
function sameModule(path1, path2) {
    // TypeScript: import paths ending in .js and .ts are resolved to the .ts file.
    // Check the base paths (without the .js or .ts suffix).
    const [basePath1, basePath2] = [path1, path2].map((p) => p.replace(/\.[tj]sx?/, ""));
    return basePath1 === basePath2 || path.resolve(basePath1) === path.resolve(basePath2);
}
exports.sameModule = sameModule;
