"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
// tslint:disable:no-console no-any
const os = require("os");
const chalk_1 = require("chalk");
const fs = require("fs");
const glob = require("glob");
const util_1 = require("util");
const cssModuleCore = require("css-modules-loader-core");
const typed_css_modules_1 = require("typed-css-modules");
const globPromise = util_1.promisify(glob);
const statPromise = util_1.promisify(fs.stat);
async function writeFile(dtsCreator, cssFile) {
    // clears cache so that watch mode generates update-to-date typing.
    const content = await dtsCreator.create(cssFile, undefined, true);
    await content.writeFile((definition) => {
        var lines = definition.trim().split(os.EOL);
        let attributes = lines.slice(1, lines.length - 2);
        attributes.sort();
        const newLines = [lines[0], ...attributes, ...lines.slice(lines.length - 2), '', ''];
        return newLines.join(os.EOL);
    });
}
async function generateTypingIfNecessary(dtsCreator, cssFile) {
    let typingStat;
    try {
        typingStat = await statPromise(cssFile + '.d.ts');
    }
    catch (_) {
        // typing does not exist: generate
        return writeFile(dtsCreator, cssFile);
    }
    const cssFileStat = await statPromise(cssFile);
    // if file is newer than typing: generate typing
    if (cssFileStat.mtime.getTime() > typingStat.mtime.getTime()) {
        return writeFile(dtsCreator, cssFile);
    }
}
class TypedCssModulesPlugin {
    constructor({ globPattern, postCssPlugins = cssModuleCore.defaultPlugins, camelCase, rootDir, searchDir, outDir, }) {
        this.useIncremental = false;
        this.globPattern = globPattern;
        if (typeof postCssPlugins === 'function') {
            postCssPlugins = postCssPlugins(cssModuleCore.defaultPlugins);
        }
        this.dtsCreator = new typed_css_modules_1.default({
            rootDir,
            searchDir,
            outDir,
            loaderPlugins: postCssPlugins,
            camelCase,
        });
    }
    apply(compiler) {
        compiler.hooks.run.tapPromise('TypedCssModulesPlugin', async () => {
            await this.generateCssTypings(this.useIncremental);
        });
        // CAVEAT: every time CSS changes, the watch-run is triggered twice:
        // - one because CSS changes
        // - one because .css.d.ts is added
        compiler.hooks.watchRun.tapPromise('TypedCssModulesPlugin', async () => {
            try {
                // first time this event is triggered, we do a full build instead of incremental build.
                await this.generateCssTypings(this.useIncremental);
            }
            catch (err) {
                console.log(chalk_1.default.bold.red(err.toString()));
            }
            finally {
                this.useIncremental = true;
            }
        });
    }
    async generateCssTypings(incremental) {
        const files = await globPromise(this.globPattern);
        const doTask = incremental ? generateTypingIfNecessary : writeFile;
        return Promise.all(files.map(file => doTask(this.dtsCreator, file)));
    }
}
exports.TypedCssModulesPlugin = TypedCssModulesPlugin;
